; File:       Decimals.asm
; Function:   Assembler routines for Decimals_asm.pas
; Language:   NASM
; Author:     Rudy Velthuis
; Copyright:  (c) 2010 Rudy Velthuis
; Version:    2.5a
; References: Donald Knuth, The Art of Computer Programming, Vol. 2:
;             Seminumerical Algorithms
; Notes:      This file contains the assembler also found in Decimals.pas, the original
;             Delphi file with built-in assembler. The assembler was transferred to this file and
;             then ported to NASM syntax.
;
; Disclaimer: Redistribution and use in source and binary forms, with or without
;             modification, are permitted provided that the following
;             conditions are met:
;
;             * Redistributions of source code must retain the above copyright
;               notice, this list of conditions and the following disclaimer.
;             * Redistributions in binary form must reproduce the above
;               copyright notice, this list of conditions and the following
;               disclaimer in the documentation and/or other materials provided
;               with the distribution.
;
;               THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER "AS IS" AND
;               ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
;               THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;               PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
;               CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
;               SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
;               NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
;               LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
;               HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
;               CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
;               OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
;               EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
;


%include "delphi.mac"

; XE is Delphi version 22.0

%assign __DELPHI_VERSION__ 22

section _BSS    public data use32 class=BSS align=4

; Converted from SysUtils.pas

%if __DELPHI_VERSION__ >= 22

; Delphi XE

record  TFormatSettings
.CurrencyString                 resd    1
                                alignb  1
.CurrencyFormat                 resb    1
                                alignb  1
.CurrencyDecimals               resb    1
                                alignb  2
.DateSeparator                  resw    1
                                alignb  2
.TimeSeparator                  resw    1
                                alignb  2
.ListSeparator                  resw    1
                                alignb  4
.ShortDateFormat                resd    1
                                alignb  4
.LongDateFormat                 resd    1
                                alignb  4
.TimeAMString                   resd    1
                                alignb  4
.TimePMString                   resd    1
                                alignb  4
.ShortTimeFormat                resd    1
                                alignb  4
.LongTimeFormat                 resd    1
                                alignb  4
.ShortMonthNames                resd    12
                                alignb  4
.LongMonthNames                 resd    12
                                alignb  4
.ShortDayNames                  resd    7
                                alignb  4
.LongDayNames                   resd    7
                                alignb  2
.ThousandSeparator              resw    1
                                alignb  2
.DecimalSeparator               resw    1
                                alignb  2
.TwoDigitYearCenturyWindow      resw    1
                                alignb  1
.NegCurrFormat                  resd    1
                                alignb  4
end;

%else

; Delphi 2010 and below

record  TFormatSettings
                                alignb  1
.CurrencyFormat                 resb    1
                                alignb  1
.NegCurrFormat                  resb    1
                                alignb  2
.ThousandSeparator              resw    1
                                alignb  2
.DecimalSeparator               resw    1
                                alignb  2
.CurrencyDecimals               resw    1
                                alignb  2
.DateSeparator                  resw    1
                                alignb  2
.TimeSeparator                  resw    1
                                alignb  4
.ListSeparator                  resd    1
                                alignb  4
.CurrencyString                 resd    1
                                alignb  4
.ShortDateFormat                resd    1
                                alignb  4
.LongDateFormat                 resd    1
                                alignb  4
.TimeAMString                   resd    1
                                alignb  4
.TimePMString                   resd    1
                                alignb  4
.ShortTimeFormat                resd    1
                                alignb  4
.LongTimeFormat                 resd    1
                                alignb  4
.ShortMonthNames                resd    12
                                alignb  4
.LongMonthNames                 resd    12
                                alignb  4
.ShortDayNames                  resd    7
                                alignb  4
.LongDayNames                   resd    7
                                alignb  1
.TwoDigitYearCenturyWindow      resb    1
                                alignb  4
end;

%endif


; Converted from Decimal_asm.pas

record  Decimal
.Lo             resd    1
.Mid            resd    1
.Hi             resd    1
.Flags          resw    1
.Scale          resb    1
.Sign           resb    1
end;

record  TAccumulator
.L0             resd    1
.L1             resd    1
.L2             resd    1
.L3             resd    1
.L4             resd    1
.L5             resd    1
.L6             resd    1
.Flags          resd    1
.Underflow      resd    1
end;

record  TAccumulator2
.Bits           resd    7
.Reserved       resw    1
.Scale          resb    1
.Sign           resb    1
end;

extern          PowersOfTen
extern          MaxMultiplicands
extern          MaxMultiplicandsMid
extern          DecimalSeparator

SingleMShift    equ     8
SingleMMask     equ     $007FFFFF
SingleEShift    equ     23
SingleEMask     equ     $0FF
SingleBias      equ     $07F

DoubleMShift    equ     11
DoubleMMask     equ     $000FFFFF
DoubleEShift    equ     52
DoubleEMask     equ     $07FF
DoubleBias      equ     $03FF

ExtendedEShift  equ     64
ExtendedEMask   equ     $7FFF
ExtendedBias    equ     $3FFF

HLW             equ     $0FFFFFFFF
HUI64           equ     $0FFFFFFFFFFFFFFFF
HLWdiv10        equ     HLW / 10
HUI64div10      equ     (HUI64 / 10) & $0FFFFFFFF

; type
;   TDecimalErrorType = (detOverflow, detUnderflow, detZeroDivide, detInvalidOp,
;     detParse, detConversion, detInvalidArg, detNaN);

enum detOverflow, detUnderflow, detZeroDivide, detInvalidOp, \
  detParse, detConversion, detInvalidArg, detNaN

TDecimalErrorType_size          equ     1
TDecimalErrorType_stacksize     equ     4


section _TEXT    public code use32 class=CODE align=16

extern  Error                   ; Decimals_nasm.Error()
extern  __UStrFromPWCharLen     ; Decimals_nasm.__UStrFromPWCharLen(), a high level wrapper
                                ; for System.@UStrFromPWCharLen
extern  Trim                    ; SysUtils.Trim()


; function DecScaleDownRaw(var D: Decimal): Byte;

function DecScaleDownRaw

asm

        PUSH    EDI

        MOV     EDI,EAX
        MOV     ECX,10
        XOR     EDX,EDX
        MOV     EAX,[EDI+Decimal.Hi]            ; [EDI].Decimal.Hi syntax not allowed in NASM
        DIV     ECX
        MOV     [EDI+Decimal.Hi],EAX
        MOV     EAX,[EDI+Decimal.Mid]
        DIV     ECX
        MOV     [EDI+Decimal.Mid],EAX
        MOV     EAX,[EDI+Decimal.Lo]
        DIV     ECX
        MOV     [EDI+Decimal.Lo],EAX
        MOV     EAX,EDX

        DEC     BYTE PTR [EDI+Decimal.Scale]

        POP     EDI
end;


; procedure DecScaleDownMaxRaw(var D: Decimal; Max: Integer);

procedure DecScaleDownMaxRaw

asm

        PUSH    EDI
        MOV     EDI,EAX

        CMP     EDX,9
        JBE     .MaxOK
        MOV     EDX,9

.MaxOK:

        SUB     [EDI+Decimal.Scale],DL
        MOV     ECX,[PowersOfTen+4*EDX]
        XOR     EDX,EDX
        MOV     EAX,[EDI+Decimal.Hi]
        DIV     ECX
        MOV     [EDI+Decimal.Hi],EAX
        MOV     EAX,[EDI+Decimal.Mid]
        DIV     ECX
        MOV     [EDI+Decimal.Mid],EAX
        MOV     EAX,[EDI+Decimal.Lo]
        DIV     ECX
        MOV     [EDI+Decimal.Lo],EAX

        SHR     ECX,1                           ; PowersOfTen[Max] div 2;
        CMP     EDX,ECX                         ; compare to Remainder
        JG      .RoundUp
        JL      .Rounded
        TEST    DWORD PTR [EDI+Decimal.Lo],1        ; if Odd(Lo) then
        JE      .Rounded                        ;   round up

.RoundUp:

        ADD     DWORD PTR [EDI+Decimal.Lo],1
        ADC     DWORD PTR [EDI+Decimal.Mid],0
        ADC     DWORD PTR [EDI+Decimal.Hi],0

.Rounded:

        POP     EDI
end;


; procedure DecScaleDownTo(var D: Decimal; TargetScale: Byte);

procedure DecScaleDownTo

asm

        PUSH    ESI
        PUSH    EBX

        MOV     ESI,EAX
        MOV     EBX,EDX
        JMP     .Loop

.TopLoop:

        MOV     EAX,ESI
        CALL    DecScaleDownRaw

.Loop:

        TEST    DWORD PTR [ESI+Decimal.Lo],1
        JNE     .Exit
        MOVZX   EAX,BYTE PTR [ESI+Decimal.Scale]
        CMP     EAX,EBX
        JLE     .Exit
        MOV     EAX,ESI
        CALL    Decimal.Mod96by5
        OR      EAX,EAX
        JE      .TopLoop

.Exit:

        POP     EBX
        POP     ESI
end;


; function DecScaleUpRaw(var D: Decimal): Longword;

function DecScaleUpRaw

asm

        PUSH    EDI
        PUSH    EBX

        MOV     EDI,EAX
        MOV     ECX,10
        MOV     EAX,[EDI+Decimal.Lo]
        MUL     ECX
        MOV     [EDI+Decimal.Lo],EAX
        MOV     EBX,EDX
        MOV     EAX,[EDI+Decimal.Mid]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+Decimal.Mid],EAX
        MOV     EBX,EDX
        MOV     EAX,[EDI+Decimal.Hi]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+Decimal.Hi],EAX
        MOV     EAX,EDX
        INC     BYTE PTR [EDI+Decimal.Scale]

        POP     EBX
        POP     EDI
end;

; function DecScaleUpMax(var D: Decimal; Max: Integer): Boolean;

procedure DecScaleUpMax

asm

        PUSH    EDI
        PUSH    EBX

        CMP     EDX,9
        JBE     .GetMaxScaleLoop
        MOV     EDX,9

.GetMaxScaleLoop:

        MOV     EDI,EAX
        MOV     ECX,DWORD PTR [MaxMultiplicands+4*EDX]
        CMP     [EDI+Decimal.Hi],ECX
        JB      .ScalesFine
        JA      .ScalesBad
        MOV     EAX,DWORD PTR [MaxMultiplicandsMid+4*EDX]
        JB      .ScalesFine

.ScalesBad:

        DEC     EDX
        JNE     .GetMaxScaleLoop
        XOR     EAX,EAX
        JMP     .Exit

.ScalesFine:

        ADD     [EDI+Decimal.Scale],DL
        MOV     ECX,DWORD PTR [PowersOfTen+4*EDX]
        MOV     EAX,[EDI+Decimal.Lo]
        MUL     ECX
        MOV     [EDI+Decimal.Lo],EAX
        MOV     EBX,EDX
        MOV     EAX,[EDI+Decimal.Mid]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+Decimal.Mid],EAX
        MOV     EBX,EDX
        MOV     EAX,[EDI+Decimal.Hi]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+Decimal.Hi],EAX
        MOV     EAX,1

.Exit:

        POP     EBX
        POP     EDI
end;


; function DecScaleUpAndAdd(var D: Decimal; Num: Byte): Longword;

function DecScaleUpAndAdd

asm

        PUSH    EDI
        PUSH    EBX

        MOV     EDI,EAX
        MOVZX   EBX,DL
        CALL    DecScaleUpRaw
        MOV     ECX,EAX
        ADD     [EDI+Decimal.Lo],EBX
        ADC     DWORD PTR [EDI+Decimal.Mid],0
        ADC     DWORD PTR [EDI+Decimal.Hi],0
        ADC     ECX,0
        MOV     EAX,ECX

        POP     EBX
        POP     EDI
end;


; procedure AccuScaleDownRaw(var A: TAccumulator);

procedure AccuScaleDownRaw

asm

        PUSH    EDI

        MOV     EDI,EAX
        MOV     ECX,10

        DEC     BYTE PTR [EDI+TAccumulator2.Scale]
        XOR     EDX,EDX
        MOV     EAX,[EDI+TAccumulator.L5]
        DIV     ECX
        MOV     [EDI+TAccumulator.L5],EAX
        MOV     EAX,[EDI+TAccumulator.L4]
        DIV     ECX
        MOV     [EDI+TAccumulator.L4],EAX
        MOV     EAX,[EDI+TAccumulator.L3]
        DIV     ECX
        MOV     [EDI+TAccumulator.L3],EAX
        MOV     EAX,[EDI+TAccumulator.L2]
        DIV     ECX
        MOV     [EDI+TAccumulator.L2],EAX
        MOV     EAX,[EDI+TAccumulator.L1]
        DIV     ECX
        MOV     [EDI+TAccumulator.L1],EAX
        MOV     EAX,[EDI+TAccumulator.L0]
        DIV     ECX
        MOV     [EDI+TAccumulator.L0],EAX
        MOV     EAX,[EDI+TAccumulator.Underflow]
        DIV     ECX
        MOV     [EDI+TAccumulator.Underflow],EAX

        POP     EDI
end;


; procedure AccuScaleDownRaw128(var A: TAccumulator);

procedure AccuScaleDownRaw128

asm

        PUSH    EDI

        MOV     EDI,EAX
        MOV     ECX,10

        DEC     BYTE PTR [EDI+TAccumulator2.Scale]
        XOR     EDX,EDX
        MOV     EAX,[EDI+TAccumulator.L3]
        DIV     ECX
        MOV     [EDI+TAccumulator.L3],EAX
        MOV     EAX,[EDI+TAccumulator.L2]
        DIV     ECX
        MOV     [EDI+TAccumulator.L2],EAX
        MOV     EAX,[EDI+TAccumulator.L1]
        DIV     ECX
        MOV     [EDI+TAccumulator.L1],EAX
        MOV     EAX,[EDI+TAccumulator.L0]
        DIV     ECX
        MOV     [EDI+TAccumulator.L0],EAX
        MOV     EAX,[EDI+TAccumulator.Underflow]
        DIV     ECX
        MOV     [EDI+TAccumulator.Underflow],EAX

        POP     EDI
end;


; procedure AccuScaleDownMax(var A: TAccumulator; Max: Integer);

procedure AccuScaleDownMax

asm

        PUSH    EDI

        MOV     EDI,EAX

        CMP     EDX,9
        JBE     .MaxOK
        MOV     EDX,9

.MaxOK:

        CMP     [EDI+TAccumulator2.Scale],DL
        JAE     .MaxScaleOK
        MOV     DL,[EDI+TAccumulator2.Scale]

.MaxScaleOK:

        MOV     ECX,DWORD PTR [PowersOfTen+4*EDX]
        SUB     [EDI+TAccumulator2.Scale],DL
        XOR     EDX,EDX

        MOV     EAX,[EDI+TAccumulator.L5]
        DIV     ECX
        MOV     [EDI+TAccumulator.L5],EAX
        MOV     EAX,[EDI+TAccumulator.L4]
        DIV     ECX
        MOV     [EDI+TAccumulator.L4],EAX
        MOV     EAX,[EDI+TAccumulator.L3]
        DIV     ECX
        MOV     [EDI+TAccumulator.L3],EAX
        MOV     EAX,[EDI+TAccumulator.L2]
        DIV     ECX
        MOV     [EDI+TAccumulator.L2],EAX
        MOV     EAX,[EDI+TAccumulator.L1]
        DIV     ECX
        MOV     [EDI+TAccumulator.L1],EAX
        MOV     EAX,[EDI+TAccumulator.L0]
        DIV     ECX
        MOV     [EDI+TAccumulator.L0],EAX
        MOV     EAX,[EDI+TAccumulator.Underflow]
        DIV     ECX
        MOV     [EDI+TAccumulator.Underflow],EAX

        POP     EDI
end;


; procedure AccuScaleDownMax128(var A: TAccumulator; Max: Integer);

procedure AccuScaleDownMax128

asm

        PUSH    EDI

        MOV     EDI,EAX

        CMP     EDX,9
        JBE     .MaxOK
        MOV     EDX,9

.MaxOK:

        CMP     [EDI+TAccumulator2.Scale],DL
        JAE     .MaxScaleOK
        MOV     DL,[EDI+TAccumulator2.Scale]

.MaxScaleOK:

        MOV     ECX,DWORD PTR [PowersOfTen+4*EDX]
        SUB     [EDI+TAccumulator2.Scale],DL
        XOR     EDX,EDX

        MOV     EAX,[EDI+TAccumulator.L3]
        DIV     ECX
        MOV     [EDI+TAccumulator.L3],EAX
        MOV     EAX,[EDI+TAccumulator.L2]
        DIV     ECX
        MOV     [EDI+TAccumulator.L2],EAX
        MOV     EAX,[EDI+TAccumulator.L1]
        DIV     ECX
        MOV     [EDI+TAccumulator.L1],EAX
        MOV     EAX,[EDI+TAccumulator.L0]
        DIV     ECX
        MOV     [EDI+TAccumulator.L0],EAX
        MOV     EAX,[EDI+TAccumulator.Underflow]
        DIV     ECX
        MOV     [EDI+TAccumulator.Underflow],EAX

        MOV     EAX,EDX

        POP     EDI
end;


; procedure AccuScaleUp(var A: TAccumulator);

procedure AccuScaleUp

asm

        PUSH    EDI
        PUSH    EBX

        MOV     EDI,EAX
        CMP     BYTE PTR [EDI+TAccumulator2.Scale],28
        JGE     .Exit
        INC     BYTE PTR [EDI+TAccumulator2.Scale]
        MOV     ECX,10

        MOV     EAX,[EDI+TAccumulator.L0]
        MUL     ECX
        MOV     [EDI+TAccumulator.L0],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L1]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L1],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L2]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L2],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L3]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L3],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L4]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L4],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L5]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L5],EAX

        MOV     EAX,EDX
.Exit:

        POP     EBX
        POP     EDI
end;


; function AccuScaleUpMax(var A: TAccumulator; Max: Integer): Longword;

function AccuScaleUpMax

asm

        PUSH    EDI
        PUSH    EBX

        MOV     EDI,EAX

        CMP     EDX,9
        JBE     .MaxOK
        MOV     EDX,9
        MOV     EAX,-1

.MaxOK:

        MOV     ECX,DWORD PTR [MaxMultiplicands+4*EDX]
        CMP     ECX,[EDI+TAccumulator.L5]
        JA      .TopLwOk
        JB      .DecrementOne
        MOV     ECX,DWORD PTR [MaxMultiplicandsMid+4*EDX]
        CMP     ECX,[EDI+TAccumulator.L4]
        JAE     .TopLwOk

.DecrementOne:

        DEC     EDX
        JE      .Exit
        JMP     .MaxOK

.TopLwOk:

        MOV     CL,28
        SUB     CL,[EDI+TAccumulator2.Scale]
        CMP     DL,CL
        JLE     .MaxScaleOK
        MOV     DL,CL

.MaxScaleOK:

        ADD     [EDI+TAccumulator2.Scale],DL
        MOV     ECX,DWORD PTR [PowersOfTen+4*EDX]

        MOV     EAX,[EDI+TAccumulator.L0]
        MUL     ECX
        MOV     [EDI+TAccumulator.L0],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L1]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L1],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L2]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L2],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L3]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L3],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L4]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L4],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L5]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L5],EAX

        MOV     EAX,EDX

.Exit:

        POP     EBX
        POP     EDI
end;


; function AccuScaleUpMax224(var A: TAccumulator; Max: Integer): Longword;

function AccuScaleUpMax224

asm

        PUSH    EDI
        PUSH    EBX

        MOV     EDI,EAX

        CMP     EDX,9
        JBE     .MaxOK
        MOV     EDX,9
        MOV     EAX,-1

.MaxOK:

        MOV     ECX,DWORD PTR [MaxMultiplicands+4*EDX]
        CMP     ECX,[EDI+TAccumulator.L6]
        JA      .TopLwOk
        JB      .DecrementOne
        MOV     ECX,DWORD PTR [MaxMultiplicandsMid+4*EDX]
        CMP     ECX,[EDI+TAccumulator.L5]
        JAE     .TopLwOk

.DecrementOne:

        DEC     EDX
        JE      .Exit
        JMP     .MaxOK

.TopLwOk:

        MOV     CL,28
        SUB     CL,[EDI+TAccumulator2.Scale]
        CMP     DL,CL
        JLE     .MaxScaleOK
        MOV     DL,CL

.MaxScaleOK:

        ADD     [EDI+TAccumulator2.Scale],DL
        MOV     ECX,DWORD PTR [PowersOfTen+4*EDX]

        MOV     EAX,[EDI+TAccumulator.L0]
        MUL     ECX
        MOV     [EDI+TAccumulator.L0],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L1]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L1],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L2]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L2],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L3]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L3],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L4]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L4],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L5]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L5],EAX
        MOV     EBX,EDX

        MOV     EAX,[EDI+TAccumulator.L6]
        MUL     ECX
        ADD     EAX,EBX
        ADC     EDX,0
        MOV     [EDI+TAccumulator.L6],EAX

        MOV     EAX,EDX

.Exit:

        POP     EBX
        POP     EDI
end;


; class function Decimal.Compare(const Left, Right: Decimal): TValueSign;

function Decimal.Compare

var     L,TAccumulator
var     R,TAccumulator

asm
        PUSH    EDI

        ; Copy Left decimal to L

        MOV     EDI,EAX
        MOV     EAX,[EDI+Decimal.Lo]
        MOV     [.L+TAccumulator.L0],EAX
        MOV     EAX,[EDI+Decimal.Mid]
        MOV     [.L+TAccumulator.L1],EAX
        MOV     EAX,[EDI+Decimal.Hi]
        MOV     [.L+TAccumulator.L2],EAX
        MOV     EAX,[EDI+Decimal.Flags]
        MOV     [.L+TAccumulator.Flags],EAX
        XOR     EAX,EAX
        MOV     [.L+TAccumulator.L3],EAX
        MOV     [.L+TAccumulator.L4],EAX
        MOV     [.L+TAccumulator.L5],EAX
        MOV     [.L+TAccumulator.Underflow],EAX

        ; Copy Right decimal to R

        MOV     EAX,[EDX+Decimal.Lo]
        MOV     [.R+TAccumulator.L0],EAX
        MOV     EAX,[EDX+Decimal.Mid]
        MOV     [.R+TAccumulator.L1],EAX
        MOV     EAX,[EDX+Decimal.Hi]
        MOV     [.R+TAccumulator.L2],EAX
        MOV     EAX,[EDX+Decimal.Flags]
        MOV     [.R+TAccumulator.Flags],EAX
        XOR     EAX,EAX
        MOV     [.R+TAccumulator.L3],EAX
        MOV     [.R+TAccumulator.L4],EAX
        MOV     [.R+TAccumulator.L5],EAX

        MOV     EAX,[.L+TAccumulator.L0]
        OR      EAX,[.L+TAccumulator.L1]
        OR      EAX,[.L+TAccumulator.L2]
        OR      EAX,[.R+TAccumulator.L0]
        OR      EAX,[.R+TAccumulator.L1]
        OR      EAX,[.R+TAccumulator.L2]
        JNE     .NotBothZero
        XOR     EAX,EAX
        JMP     .Finish

.NotBothZero:

        MOV     AL,[.L+TAccumulator2.Sign]
        CMP     AL,[.R+TAccumulator2.Sign]
        JE      .LeftScaleLoop
        JA      .Below
        MOV     EAX,1
        JMP     .Finish

.Below:

        MOV     EAX,-1
        JMP     .Finish

.LeftScaleLoop:

        ; If necessary, scale L up

        MOV     AL,[.R+TAccumulator2.Scale]
        SUB     AL,[.L+TAccumulator2.Scale]
        JE      .DoComparison
        JL      .RightScaleLoop
        MOVZX   EDX,AL
        LEA     EAX,[.L]
        CALL    AccuScaleUpMax
        JMP     .LeftScaleLoop

.RightScaleLoop:

        ; If necessary, scale R up

        MOV     AL,[.L+TAccumulator2.Scale]
        SUB     AL,[.R+TAccumulator2.Scale]
        JE      .DoComparison
        MOVZX   EDX,AL
        LEA     EAX,[.R]
        CALL    AccuScaleUpMax
        JMP     .RightScaleLoop

.DoComparison:

        ; Compare L and R, top down

        MOV     EAX,1
        MOV     EDX,[.L+TAccumulator.L5]
        CMP     EDX,[.R+TAccumulator.L5]
        JA      .Greater
        JB      .Less
        MOV     EDX,[.L+TAccumulator.L4]
        CMP     EDX,[.R+TAccumulator.L4]
        JA      .Greater
        JB      .Less
        MOV     EDX,[.L+TAccumulator.L3]
        CMP     EDX,[.R+TAccumulator.L3]
        JA      .Greater
        JB      .Less
        MOV     EDX,[.L+TAccumulator.L2]
        CMP     EDX,[.R+TAccumulator.L2]
        JA      .Greater
        JB      .Less
        MOV     EDX,[.L+TAccumulator.L1]
        CMP     EDX,[.R+TAccumulator.L1]
        JA      .Greater
        JB      .Less
        MOV     EDX,[.L+TAccumulator.L0]
        CMP     EDX,[.R+TAccumulator.L0]
        JA      .Greater
        JB      .Less
        JMP     .Equal

.Less:

        DEC     EAX

.Equal:

        DEC     EAX

.Greater:

        CMP     BYTE PTR [.L+TAccumulator2.Sign],0
        JE      .Finish
        NEG     EAX

.Finish:

        POP     EDI

end;


; class procedure Decimal.InternalFromDouble(out Result: Decimal; const Source: Double);

procedure Decimal.InternalFromDouble

param   Source,Double           ; can be accessed as [.Source]

var     A,TAccumulator          ; can be accessed as [.A]
var     LResult,Pointer         ; can be accessed as [.LResult]

asm
        PUSH    EDI
        PUSH    ESI
        PUSH    EBX
        MOV     [.LResult],EAX

        XOR     EAX,EAX
        MOV     [.A+TAccumulator.L0],EAX
        MOV     [.A+TAccumulator.L1],EAX
        MOV     [.A+TAccumulator.L2],EAX
        MOV     [.A+TAccumulator.L3],EAX
        MOV     [.A+TAccumulator.L4],EAX
        MOV     [.A+TAccumulator.L5],EAX
        MOV     [.A+TAccumulator.Flags],EAX

        MOV     ESI,DWORD PTR [.Source]
        MOV     EBX,DWORD PTR [.Source+4]
        MOV     EDX,EBX
        AND     EBX,DoubleMMask
        OR      EBX,DoubleMMask+1
        SHLD    EBX,ESI,DoubleMShift
        SHL     ESI,DoubleMShift        ; mantissa in EBX:ESI

        SHR     EDX,DoubleEShift
        TEST    EDX,DoubleEMask+1
        JE      .Positive
        OR      BYTE PTR [.A+TAccumulator2.Sign],$080

.Positive:

        AND     EDX,DoubleEMask
        JE      .Finish
        CMP     EDX,DoubleEMask
        JNE     .NoSpecialValue
        CMP     EBX,$080000000
        JNE     .NaN
        OR      ESI,ESI
        JE      .Infinite

.NaN:

        MOV     EAX,detNaN
        JMP     Error

.Infinite:

        MOV     EAX,$0FFFFFFFF
        MOV     [.A+TAccumulator.L3],EAX
        MOV     [.A+TAccumulator.L4],EAX
        MOV     [.A+TAccumulator.L5],EAX
        JMP     .Finish

.NoSpecialValue:

        SUB     EDX,DoubleBias-96
        JNS     .NoUnderflow
        MOV     EAX,detUnderflow
        JMP     Error

.NoUnderflow:

        CMP     EDX,191
        JBE     .NoOverflow
        MOV     EAX,detOverflow
        JMP     Error

.NoOverflow:

        MOV     CL,DL
        NOT     CL
        XOR     EAX,EAX
        SHR     EDX,5
        SHRD    EAX,ESI,CL
        SHRD    ESI,EBX,CL
        SHR     EBX,CL                  ; Shifted mantissa in EBX:ESI:EAX

        MOV     DWORD PTR [.A+4*EDX],EBX
        DEC     EDX
        JNS     .StoreESI
        CMP     ESI,$080000000
        JB      .ScaleLoop
        JA      .DoRound0
        TEST    EBX,1
        JE      .ScaleLoop

.DoRound0:

        ADD     DWORD PTR [.A+TAccumulator.L0],1
        ADC     DWORD PTR [.A+TAccumulator.L1],0
        JMP     .ScaleLoop

.StoreESI:

        MOV     DWORD PTR [.A+4*EDX],ESI
        DEC     EDX
        JNS     .StoreEAX
        CMP     EAX,$080000000
        JB      .ScaleLoop
        JA      .DoRound1
        TEST    ESI,1
        JE      .ScaleLoop

.DoRound1:

        ADD     DWORD PTR [.A+TAccumulator.L0],1
        ADC     DWORD PTR [.A+TAccumulator.L1],0
        ADC     DWORD PTR [.A+TAccumulator.L2],0
        JMP     .ScaleLoop

.StoreEAX:

        MOV     [.A+4*EDX],EAX

.ScaleLoop:

        CMP     BYTE PTR [.A+TAccumulator2.Scale],28
        JGE     .Finish
        MOV     EAX,[.A+TAccumulator.L0]
        OR      EAX,[.A+TAccumulator.L1]
        OR      EAX,[.A+TAccumulator.L2]
        JE      .Finish
        CMP     DWORD PTR [.A+TAccumulator.L5],HLWdiv10
        JA      .Finish
        LEA     EAX,[.A]
        CALL    AccuScaleUp
        JMP     .ScaleLoop

.Finish:

        MOV     EDI,[.LResult]
        MOV     EAX,[.A+TAccumulator.L3]
        MOV     [EDI+Decimal.Lo],EAX
        MOV     EAX,[.A+TAccumulator.L4]
        MOV     [EDI+Decimal.Mid],EAX
        MOV     EAX,[.A+TAccumulator.L5]
        MOV     [EDI+Decimal.Hi],EAX
        MOV     EAX,[.A+TAccumulator.Flags]
        MOV     [EDI+Decimal.Flags],EAX

        POP     EBX
        POP     ESI
        POP     EDI

end;


; class procedure Decimal.InternalFromExtended(out Result: Decimal; const Source: Extended);

procedure Decimal.InternalFromExtended

param   Source,Extended

var     A,TAccumulator
var     LResult,Pointer

asm
        PUSH    ESI
        PUSH    EDI
        PUSH    EBX
        MOV     [.LResult],EAX

        XOR     EAX,EAX
        MOV     [.A+TAccumulator.L0],EAX
        MOV     [.A+TAccumulator.L1],EAX
        MOV     [.A+TAccumulator.L2],EAX
        MOV     [.A+TAccumulator.L3],EAX
        MOV     [.A+TAccumulator.L4],EAX
        MOV     [.A+TAccumulator.L5],EAX
        MOV     [.A+TAccumulator.Flags],EAX
        MOV     EDI,DWORD PTR [.Source]
        MOV     ESI,DWORD PTR [.Source+4]
        MOVZX   EDX,WORD PTR [.Source+8]
        MOV     CH,DH
        AND     CH,$80
        MOV     [.A+TAccumulator2.Sign],CH
        AND     EDX,ExtendedEMask
        JE      .Finish
        CMP     EDX,ExtendedEMask
        JNE     .NoSpecialValue
        CMP     ESI,$80000000
        JNE     .NaN
        OR      EDI,EDI
        JE      .Infinite

.NaN:

        MOV     EAX,detNaN
        CALL    Error

.Infinite:

        MOV     EAX,-1
        MOV     [.A+TAccumulator.L3],EAX
        MOV     [.A+TAccumulator.L4],EAX
        MOV     [.A+TAccumulator.L5],EAX
        JMP     .Finish

.NoSpecialValue:

        SUB     EDX,ExtendedBias-97
        CMP     EDX,$000000BF
        JBE     .NoOverflow
        MOV     EAX,detOverflow
        JMP     Error

.NoOverflow:

        CMP     EDX,0
        JGE     .NoUnderflow
        MOV     EAX,detUnderflow
        JMP     Error

.NoUnderflow:

        XOR     EBX,EBX
        MOV     ECX,EDX
        SHR     EDX,5
        XOR     EAX,EAX
        NEG     ECX
        AND     ECX,$1F
        JE      .ZeroShift
        SHRD    EBX,EDI,CL
        SHRD    EDI,ESI,CL
        SHR     ESI,CL
        JMP     .Shifted

.ZeroShift:

        DEC     EDX

.Shifted:

        MOV     DWORD PTR [.A+4*EDX],ESI
        DEC     EDX
        JS      .ScaleUpLoop
        MOV     DWORD PTR [.A+4*EDX],EDI
        DEC     EDX
        JS      .ScaleUpLoop
        MOV     DWORD PTR [.A+4*EDX],EBX

.ScaleUpLoop:

        MOV     EAX,[.A+TAccumulator.L0]
        OR      EAX,[.A+TAccumulator.L1]
        OR      EAX,[.A+TAccumulator.L2]
        JE      .Finish
        CMP     BYTE PTR [.A+TAccumulator2.Scale],28
        JAE     .Finish
        CMP     DWORD PTR [.A+TAccumulator.L5],HLWdiv10
        JA      .Finish
        LEA     EAX,[.A]
        CALL    AccuScaleUp
        JMP     .ScaleUpLoop

.Finish:

        MOV     EDI,[.LResult]
        MOV     EAX,[.A+TAccumulator.L3]
        MOV     [EDI+Decimal.Lo],EAX
        MOV     EAX,[.A+TAccumulator.L4]
        MOV     [EDI+Decimal.Mid],EAX
        MOV     EAX,[.A+TAccumulator.L5]
        MOV     [EDI+Decimal.Hi],EAX
        MOV     EAX,[.A+TAccumulator.Flags]
        MOV     [EDI+Decimal.Flags],EAX

        POP     EBX
        POP     EDI
        POP     ESI
end;


; class procedure Decimal.InternalFromSingle(out Result: Decimal; const Source: Single);

procedure Decimal.InternalFromSingle

param   Source,Single

var     A,TAccumulator
var     LResult,Pointer

asm
        PUSH    ESI
        PUSH    EDI
        MOV     [.LResult],EAX

        XOR     EAX,EAX
        MOV     [.A+ \
        TAccumulator.L0],EAX
        MOV     [.A+TAccumulator.L1],EAX
        MOV     [.A+TAccumulator.L2],EAX
        MOV     [.A+TAccumulator.L3],EAX
        MOV     [.A+TAccumulator.L4],EAX
        MOV     [.A+TAccumulator.L5],EAX
        MOV     [.A+TAccumulator.Flags],EAX

        MOV     ESI,[.Source]
        MOV     EDX,ESI
        AND     ESI,SingleMMask         ;// 23 bit stored mantissa
        OR      ESI,SingleMMask+1       ;// set hidden bit
        SHL     ESI,SingleMShift        ;// shift up 8 bits

        SHR     EDX,SingleEShift
        OR      DH,DH
        JE      .Positive
        OR      BYTE PTR [.A+TAccumulator2.Sign],$080
        XOR     DH,DH

.Positive:
        OR      DL,DL

        JE      .Finish
        CMP     DL,$0FF
        JNE     .NoSpecialValue
        CMP     ESI,$080000000
        JE      .Infinite
        MOV     EAX,detNaN
        JMP     Error

.Infinite:

        MOV     EAX,-1
        MOV     [.A+TAccumulator.L3],EAX
        MOV     [.A+TAccumulator.L4],EAX
        MOV     [.A+TAccumulator.L5],EAX
        JMP     .Finish

.NoSpecialValue:

        SUB     DL,SingleBias-96
        MOV     CL,DL
        NOT     CL

        XOR     EAX,EAX
        SHR     EDX,5
        SHRD    EAX,ESI,CL     ;// ESI: top bits
        SHR     ESI,CL         ;// EAX: bottom bits

        MOV     DWORD PTR [.A+4*EDX],ESI
        DEC     EDX
        JNS     .StoreEAX
        CMP     EAX,$80000000
        JB      .InPosition
        JA      .DoRound
        TEST    ESI,1
        JE      .InPosition

.DoRound:

        ADD     DWORD PTR [.A+TAccumulator.L0],1
        ADC     DWORD PTR [.A+TAccumulator.L1],0
        JMP     .InPosition

.StoreEAX:

        MOV     DWORD PTR [.A+4*EDX],EAX

.InPosition:

        CMP     BYTE PTR [.A+TAccumulator2.Scale],28
        JGE     .Finish
        MOV     EAX,[.A+TAccumulator.L0]
        OR      EAX,[.A+TAccumulator.L1]
        OR      EAX,[.A+TAccumulator.L2]
        JE      .Finish
        LEA     EAX,[.A]
        CALL    AccuScaleUp
        JMP     .InPosition

.Finish:

        MOV     EDI,[.LResult]
        MOV     EAX,[.A+TAccumulator.L3]
        MOV     [EDI+Decimal.Lo],EAX
        MOV     EAX,[.A+TAccumulator.L4]
        MOV     [EDI+Decimal.Mid],EAX
        MOV     EAX,[.A+TAccumulator.L5]
        MOV     [EDI+Decimal.Hi],EAX
        MOV     EAX,[.A+TAccumulator.Flags]
        MOV     [EDI+Decimal.Flags],EAX

        POP     EDI
        POP     ESI
end;


; class procedure Decimal.InternalAddSub(var Result: Decimal; const Left, Right: Decimal;
;   Sign: Byte);

procedure Decimal.InternalAddSub

param   ASign,Byte

var     L,TAccumulator
var     R,TAccumulator

asm
        PUSH    EBX

        MOV     EBX,EAX
        XOR     EAX,EAX

        ;// Copy Left decimal into L

        MOV     EAX,[EDX+Decimal.Lo]
        MOV     [.L+TAccumulator.L0],EAX
        MOV     EAX,[EDX+Decimal.Mid]
        MOV     [.L+TAccumulator.L1],EAX
        MOV     EAX,[EDX+Decimal.Hi]
        MOV     [.L+TAccumulator.L2],EAX
        MOV     EAX,[EDX+Decimal.Flags]
        MOV     [.L+TAccumulator.Flags],EAX
        XOR     EAX,EAX
        MOV     [.L+TAccumulator.L3],EAX
        MOV     [.L+TAccumulator.L4],EAX
        MOV     [.L+TAccumulator.L5],EAX
        MOV     [.L+TAccumulator.Underflow],EAX

        ;// Copy Right decimal into R

        MOV     EAX,[ECX+Decimal.Lo]
        MOV     [.R+TAccumulator.L0],EAX
        MOV     EAX,[ECX+Decimal.Mid]
        MOV     [.R+TAccumulator.L1],EAX
        MOV     EAX,[ECX+Decimal.Hi]
        MOV     [.R+TAccumulator.L2],EAX
        MOV     EAX,[ECX+Decimal.Flags]
        MOV     [.R+TAccumulator.Flags],EAX
        MOV     AL,[.ASign]
        XOR     [.R+TAccumulator2.Sign],AL
        XOR     EAX,EAX
        MOV     [.R+TAccumulator.L3],EAX
        MOV     [.R+TAccumulator.L4],EAX
        MOV     [.R+TAccumulator.L5],EAX

        ;// If necessary, scale L up to match scale of R

        MOV     AL,[.L+TAccumulator2.Scale]
        SUB     AL,[.R+TAccumulator2.Scale]
        JE      .CheckSigns
        JA      .ScaleRightLoop
        NEG     AL

.ScaleLeftLoop:

        MOVZX   EDX,AL
        LEA     EAX,[.L]
        CALL    AccuScaleUpMax
        MOV     AL,[.R+TAccumulator2.Scale]
        SUB     AL,[.L+TAccumulator2.Scale]
        JNE     .ScaleLeftLoop
        JMP     .CheckSigns

.ScaleRightLoop:

        MOVZX   EDX,AL
        LEA     EAX,[.R]
        CALL    AccuScaleUpMax
        MOV     AL,[.L+TAccumulator2.Scale]
        SUB     AL,[.R+TAccumulator2.Scale]
        JNE     .ScaleRightLoop

.CheckSigns:

        MOV     AL,[.L+TAccumulator2.Sign]
        XOR     AL,[.R+TAccumulator2.Sign]
        JNE     .DoSubtract

        ;// L := L + R

        MOV     EAX,[.R+TAccumulator.L0]
        ADD     [.L+TAccumulator.L0],EAX
        MOV     EAX,[.R+TAccumulator.L1]
        ADC     [.L+TAccumulator.L1],EAX
        MOV     EAX,[.R+TAccumulator.L2]
        ADC     [.L+TAccumulator.L2],EAX
        MOV     EAX,[.R+TAccumulator.L3]
        ADC     [.L+TAccumulator.L3],EAX
        MOV     EAX,[.R+TAccumulator.L4]
        ADC     [.L+TAccumulator.L4],EAX
        MOV     EAX,[.R+TAccumulator.L5]
        ADC     [.L+TAccumulator.L5],EAX
        JMP     .ScaleDownLoop

.DoSubtract:

        ;// L := L - R.

        MOV     EAX,[.R+TAccumulator.L0]
        SUB     [.L+TAccumulator.L0],EAX
        MOV     EAX,[.R+TAccumulator.L1]
        SBB     [.L+TAccumulator.L1],EAX
        MOV     EAX,[.R+TAccumulator.L2]
        SBB     [.L+TAccumulator.L2],EAX
        MOV     EAX,[.R+TAccumulator.L3]
        SBB     [.L+TAccumulator.L3],EAX
        MOV     EAX,[.R+TAccumulator.L4]
        SBB     [.L+TAccumulator.L4],EAX
        MOV     EAX,[.R+TAccumulator.L5]
        SBB     [.L+TAccumulator.L5],EAX
        JNC     .ScaleDownLoop

        ;// if L < 0 then L := -L;

        XOR     BYTE PTR [.L+TAccumulator2.Sign],$80
        XOR     EDX,EDX
        MOV     EAX,EDX
        SUB     EAX,[.L+TAccumulator.L0]
        MOV     [.L+TAccumulator.L0],EAX
        MOV     EAX,EDX
        SBB     EAX,[.L+TAccumulator.L1]
        MOV     [.L+TAccumulator.L1],EAX
        MOV     EAX,EDX
        SBB     EAX,[.L+TAccumulator.L2]
        MOV     [.L+TAccumulator.L2],EAX
        MOV     EAX,EDX
        SBB     EAX,[.L+TAccumulator.L3]
        MOV     [.L+TAccumulator.L3],EAX
        MOV     EAX,EDX
        SBB     EAX,[.L+TAccumulator.L4]
        MOV     [.L+TAccumulator.L4],EAX
        MOV     EAX,EDX
        SBB     EAX,[.L+TAccumulator.L5]
        MOV     [.L+TAccumulator.L5],EAX

        ;// Scale down until top 3 longwords are 0.

.ScaleDownLoop:

        CMP     DWORD PTR [.L+TAccumulator.L5],0
        JE      .CheckL4
        LEA     EAX,[.L]
        MOV     EDX,9
        CALL    AccuScaleDownMax
        JMP     .ScaleDownLoop

.CheckL4:

        CMP     DWORD PTR [.L+TAccumulator.L4],0
        JE      .CheckL3
        LEA     EAX,[.L]
        MOV     EDX,9
        CALL    AccuScaleDownMax
        JMP     .CheckL4

.CheckL3:

        CMP     DWORD PTR [.L+TAccumulator.L3],0
        JNE     .ScaleDown
        CMP     BYTE PTR [.L+TAccumulator2.Scale],28
        JBE     .Finish

.ScaleDown:

        CMP     BYTE PTR [.L+TAccumulator2.Scale],0
        JLE     .Finish
        LEA     EAX,[.L]
        CALL    AccuScaleDownRaw
        JMP     .CheckL3

        ;// Check for overflow. The above loop is aborted if Scale outside its
        ;// range, so the top 3 longwords can still be <> 0.
.Finish:

        CMP     DWORD PTR [.L+TAccumulator.Underflow],$80000000
        JB      .NoRound
        JA      .DoRound
        CMP     DWORD PTR [.L+TAccumulator.L0],1
        JZ      .NoRound

.DoRound:

        ADD     DWORD PTR [.L+TAccumulator.L0],1
        ADC     DWORD PTR [.L+TAccumulator.L1],0
        ADC     DWORD PTR [.L+TAccumulator.L2],0

.NoRound:

        MOV     EAX,[.L+TAccumulator.L5]
        OR      EAX,[.L+TAccumulator.L4]
        OR      EAX,[.L+TAccumulator.L3]
        JE      .Exit
        MOV     EAX,detOverflow
        CALL    Error

        ;// Copy Flags:L2:L1:L0 into Result.
.Exit:

        MOV     EAX,[.L+TAccumulator.L0]
        MOV     [EBX+Decimal.Lo],EAX
        MOV     EAX,[.L+TAccumulator.L1]
        MOV     [EBX+Decimal.Mid],EAX
        MOV     EAX,[.L+TAccumulator.L2]
        MOV     [EBX+Decimal.Hi],EAX
        MOV     EAX,[.L+TAccumulator.Flags]
        MOV     [EBX+Decimal.Flags],EAX

        POP     EBX
end;

; class procedure Decimal.InternalCeiling(var Result: Decimal; const Source: Decimal);

procedure Decimal.InternalCeiling

asm
        PUSH    EDI

        MOV     EDI,EAX
        CALL    Decimal.InternalTruncate
        CMP     BYTE PTR [EDI+Decimal.Sign],0
        JNE     .Exit
        ADD     DWORD PTR [EDI+Decimal.Lo],1
        ADC     DWORD PTR [EDI+Decimal.Mid],0
        ADC     DWORD PTR [EDI+Decimal.Hi],0
.Exit:
        POP     EDI
end;


; class function Decimal.Div192by32(Quotient: PLongword; const Dividend: Decimal; Divisor: Longword): Longword;

function Decimal.Div192by32

asm

        PUSH    ESI
        PUSH    EDI

        MOV     EDI,EAX                 ;// EDI is Quotient
        MOV     ESI,EDX                 ;// ESI is Dividend
        CMP     ECX,0
        JNE     .NoZeroDivide32

        MOV     EAX,detZeroDivide
        JMP     Error

.NoZeroDivide32:

        XOR     EDX,EDX
        MOV     EAX,[ESI+Decimal.Hi]
        DIV     ECX
        MOV     [EDI+TAccumulator.L6],EAX
        MOV     EAX,[ESI+Decimal.Mid]
        DIV     ECX
        MOV     [EDI+TAccumulator.L5],EAX
        MOV     EAX,[ESI+Decimal.Lo]
        DIV     ECX
        MOV     [EDI+TAccumulator.L4],EAX
        MOV     EAX,0
        DIV     ECX
        MOV     [EDI+TAccumulator.L3],EAX
        MOV     EAX,0
        DIV     ECX
        MOV     [EDI+TAccumulator.L2],EAX
        MOV     EAX,0
        DIV     ECX
        MOV     [EDI+TAccumulator.L1],EAX
        MOV     EAX,0
        DIV     ECX
        MOV     [EDI+TAccumulator.L0],EAX

        MOV     EAX,EDX

        POP     EDI
        POP     ESI
end;


;// Basecase division
;//
;// The following two routines use Donald Knuth's algorithm D (Vol. 2, section 4.3.1) to do a
;// faster division. It works (almost) like a normal long division, as taught in many schools,
;// but unlike a binary long division, which has to loop once for every bit, it finds an entire
;// digit (longword) in one pass. There are m - n + 1 passes, where m is the number of digits
;// (or "limbs") of the dividend, and n the number of digits of the divisor.
;//
;// For this to work, the divisor must be "normalized", i.e. it is shifted left until the top bit
;// is set. The dividend is shifted left accordingly, so the result remains the same.
;//
;// Just like in normal long division done manually, the divisor is aligned under the dividend,
;// and a guess for the first result digit is made. In this algorithm this is achieved by dividing
;// the top two digits of the dividend (or the remainder of it) by the top digit of the divisor. The
;// guess can of course be too big or too high, and some adjustments are made. Then divisor*digit
;// is subtracted from the aligned digits of the dividend (or its remainder). If the result is
;// negative, the divisor is added back once and the estimated digit is decremented. Now the digit
;// can be stored, and the loop repeats with the remainder of the dividend.

; class function Decimal.Div192by64(Quotient: PLongword; const Dividend, Divisor: Decimal): Longword;

procedure Decimal.Div192by64

var     LDivisor,Decimal
var     LDividend,8*Longword_size       ; array[0..7] of Longword
var     QHat,UInt64
var     RHat,Longword
var     A,TAccumulator

asm

        PUSH    ESI
        PUSH    EDI
        PUSH    EBX

        MOV     EDI,EAX                 ;// EDI = Quotient
        MOV     ESI,EDX                 ;// ESI = Dividend
        MOV     EBX,ECX                 ;// EBX = Divisor

        ;// Top Longword of Divisor is 0, so ignore it.

        MOV     EAX,[EBX+Decimal.Mid]
        BSR     ECX,EAX
        SUB     ECX,31
        NEG     ECX
        JE      .NoShift64

        MOV     EDX,[EBX+Decimal.Lo]
        SHLD    EAX,EDX,CL
        MOV     [.LDivisor+Decimal.Mid],EAX
        SHL     EDX,CL
        MOV     [.LDivisor+Decimal.Lo],EDX

        XOR     EAX,EAX
        MOV     EDX,[ESI+Decimal.Hi]
        SHLD    EAX,EDX,CL
        MOV     DWORD PTR [.LDividend+28],EAX
        MOV     EAX,[ESI+Decimal.Mid]
        SHLD    EDX,EAX,CL
        MOV     DWORD PTR [.LDividend+24],EDX
        MOV     EDX,[ESI+Decimal.Lo]
        SHLD    EAX,EDX,CL
        MOV     DWORD PTR [.LDividend+20],EAX
        SHL     EDX,CL
        MOV     DWORD PTR [.LDividend+16],EDX
        JMP     .PrepareMainLoop64

.NoShift64:

        MOV     EAX,[EBX+Decimal.Mid]
        MOV     [.LDivisor+Decimal.Mid],EAX
        MOV     EAX,[EBX+Decimal.Lo]
        MOV     [.LDivisor+Decimal.Lo],EAX

        MOV     DWORD PTR [.LDividend+28],0
        MOV     EAX,[ESI+Decimal.Hi]
        MOV     DWORD PTR [.LDividend+24],EAX
        MOV     EAX,[ESI+Decimal.Mid]
        MOV     DWORD PTR [.LDividend+20],EAX
        MOV     EAX,[ESI+Decimal.Lo]
        MOV     DWORD PTR [.LDividend+16],EAX

.PrepareMainLoop64:

        XOR     EAX,EAX
        MOV     [.LDivisor+Decimal.Hi],EAX
        MOV     DWORD PTR [.LDividend+12],EAX
        MOV     DWORD PTR [.LDividend+8],EAX
        MOV     DWORD PTR [.LDividend+4],EAX
        MOV     DWORD PTR [.LDividend],EAX
        MOV     [EDI+TAccumulator.L6],EAX
        MOV     [EDI+TAccumulator.L5],EAX
        MOV     [EDI+TAccumulator.L4],EAX
        MOV     [EDI+TAccumulator.L3],EAX
        MOV     [EDI+TAccumulator.L2],EAX
        MOV     [EDI+TAccumulator.L1],EAX
        MOV     [EDI+TAccumulator.L0],EAX

        MOV     ECX,5           ;// m - n + 1 = 7 - 2 + 1 = 6 --> 0..5

.MainLoop64:

        XOR     EDX,EDX
        MOV     EBX,[.LDivisor+Decimal.Mid]
        MOV     EAX,DWORD PTR [.LDividend+4*ECX+8]
        DIV     EBX
        MOV     [.QHat+4],EAX
        MOV     EAX,DWORD PTR [.LDividend+4*ECX+4]
        MOV     ESI,EAX
        DIV     EBX
        MOV     [.QHat],EAX


        MUL     EBX
        MOV     [.A+TAccumulator.L0],EAX
        MOV     [.A+TAccumulator.L1],EDX
        MOV     EAX,[.QHat+4]
        MUL     EBX
        ADD     EAX,[.A+TAccumulator.L1]
        SUB     ESI,[.A+TAccumulator.L0]
        MOV     [.RHat],ESI

.AdjustLoop64:

        CMP     DWORD PTR [.QHat+4],0
        JNE     .DoAdjust64
        MOV     EBX,[.LDivisor+Decimal.Lo]
        MOV     EAX,[.QHat]
        MUL     EBX
        SUB     EAX,DWORD PTR [.LDividend+4*ECX]
        SBB     EDX,[.RHat]
        JBE     .AdjustLoopEnd64

.DoAdjust64:

        ;// This part is called at most 2 times (see Knuth).

        SUB     DWORD PTR [.QHat],1
        SBB     DWORD PTR [.QHat+4],0
        MOV     EAX,[.LDivisor+Decimal.Mid]
        ADD     [.RHat],EAX
        JZ      .AdjustLoop64

.AdjustLoopEnd64:

        MOV     EBX,[.QHat]
        MOV     EAX,[.LDivisor+Decimal.Lo]
        MUL     EBX
        MOV     DWORD PTR [.A+TAccumulator.L0],EAX
        MOV     ESI,EDX
        MOV     EAX,[.LDivisor+Decimal.Mid]
        MUL     EBX
        ADD     EAX,ESI
        ADC     EDX,0
        MOV     [.A+TAccumulator.L1],EAX
        MOV     EAX,[.A+TAccumulator.L0]
        SUB     [.LDividend+4*ECX],EAX
        MOV     EAX,[.A+TAccumulator.L1]
        SBB     [.LDividend+4*ECX+4],EAX
        SBB     [.LDividend+4*ECX+8],EDX

        MOV     [EDI+4*ECX],EBX

        JNC     .EstimateCorrect64

        DEC     DWORD PTR [EDI+4*ECX]
        MOV     EAX,[.LDivisor+Decimal.Lo]
        ADD     [.LDividend+4*ECX],EAX
        MOV     EAX,[.LDivisor+Decimal.Mid]
        ADC     [.LDividend+4*ECX+4],EAX
        ADC     DWORD PTR [.LDividend+4*ECX+8],0

.EstimateCorrect64:

        SUB     ECX,1
        JNC     .MainLoop64

.Exit64:

        MOV     EAX,[.RHat]

        POP     EBX
        POP     EDI
        POP     ESI
end;


; class function Decimal.Div192by96(Quotient: PLongword; const Dividend, Divisor: Decimal): Longword;

function Decimal.Div192by96

var LDivisor,Decimal
var LDividend,8*Longword_size
var QHat,UInt64
var RHat,Longword
var A,TAccumulator

asm

        PUSH    ESI
        PUSH    EDI
        PUSH    EBX

        MOV     EDI,EAX
        MOV     ESI,EDX
        MOV     EBX,ECX

        ;// Determine number of leading zero bits of divisor and store it in CL.

        MOV     EAX,[EBX+Decimal.Hi]
        BSR     ECX,EAX
        SUB     ECX,31
        NEG     ECX
        JE      .NoShift96

        ;// Shift divisor left by CL and store in LDivisor.

        MOV     EDX,[EBX+Decimal.Mid]
        SHLD    EAX,EDX,CL
        MOV     [.LDivisor+Decimal.Hi],EAX
        MOV     EAX,[EBX+Decimal.Lo]
        SHLD    EDX,EAX,CL
        MOV     [.LDivisor+Decimal.Mid],EDX
        SHL     EAX,CL
        MOV     [.LDivisor+Decimal.Lo],EAX

        ;// Shift dividend left by same number of bits (CL) and store in LDividend.

        XOR     EAX,EAX
        MOV     EDX,[ESI+Decimal.Hi]
        SHLD    EAX,EDX,CL
        MOV     DWORD PTR [.LDividend+28],EAX
        MOV     EAX,[ESI+Decimal.Mid]
        SHLD    EDX,EAX,CL
        MOV     DWORD PTR [.LDividend+24],EDX
        MOV     EDX,[ESI+Decimal.Lo]
        SHLD    EAX,EDX,CL
        MOV     DWORD PTR [.LDividend+20],EAX
        SHL     EDX,CL
        MOV     DWORD PTR [.LDividend+16],EDX
        JMP     .PrepareMainLoop96

.NoShift96:

        ;// No shift required, so simply copy.

        MOV     EAX,[EBX+Decimal.Hi]
        MOV     [.LDivisor+Decimal.Hi],EAX
        MOV     EAX,[EBX+Decimal.Mid]
        MOV     [.LDivisor+Decimal.Mid],EAX
        MOV     EAX,[EBX+Decimal.Lo]
        MOV     [.LDivisor+Decimal.Lo],EAX

        MOV     DWORD PTR [.LDividend+28],0
        MOV     EAX,[ESI+Decimal.Hi]
        MOV     DWORD PTR [.LDividend+24],EAX
        MOV     EAX,[ESI+Decimal.Mid]
        MOV     DWORD PTR [.LDividend+20],EAX
        MOV     EAX,[ESI+Decimal.Lo]
        MOV     DWORD PTR [.LDividend+16],EAX

.PrepareMainLoop96:

        ;// Zero out certain parts.

        XOR     EAX,EAX
        MOV     DWORD PTR [.LDividend],EAX
        MOV     DWORD PTR [.LDividend+4],EAX
        MOV     DWORD PTR [.LDividend+8],EAX
        MOV     DWORD PTR [.LDividend+12],EAX
        MOV     [EDI+TAccumulator.L0],EAX
        MOV     [EDI+TAccumulator.L1],EAX
        MOV     [EDI+TAccumulator.L2],EAX
        MOV     [EDI+TAccumulator.L3],EAX
        MOV     [EDI+TAccumulator.L4],EAX
        MOV     [EDI+TAccumulator.L5],EAX
        MOV     [EDI+TAccumulator.L6],EAX

        ;// for i := 3 downto 0 do ...

        MOV     ECX,4

.MainLoop96:

        ;// Divide top two digits of current part of dividend (N) by top digit
        ;// of divisor (D) = QHat
        ;// IOW: QHat := N div D;

        XOR     EDX,EDX
        MOV     EBX,[.LDivisor+Decimal.Hi]
        MOV     EAX,DWORD PTR [.LDividend+4*ECX+12]
        DIV     EBX
        MOV     [.QHat+4],EAX
        MOV     EAX,DWORD PTR [.LDividend+4*ECX+8]
        MOV     ESI,EAX
        DIV     EBX
        MOV     [.QHat],EAX

        ;// .RHat := N - .QHat * D;

        MUL     EBX
        MOV     [.A+TAccumulator.L0],EAX
        MOV     [.A+TAccumulator.L1],EDX
        MOV     EAX,[.QHat+4]
        MUL     EBX
        ADD     EAX,[.A+TAccumulator.L1]
        SUB     ESI,[.A+TAccumulator.L0]
        MOV     [.RHat],ESI

.AdjustLoop96:

        CMP     DWORD PTR [.QHat+4],0
        JNE     .DoAdjust96
        MOV     EBX,[.LDivisor+Decimal.Mid]
        MOV     EAX,[.QHat]
        MUL     EBX
        SUB     EAX,DWORD PTR [.LDividend+4*ECX+4]
        SBB     EDX,[.RHat]
        JBE     .AdjustLoopEnd96

.DoAdjust96:

        SUB     DWORD PTR [.QHat],1
        SBB     DWORD PTR [.QHat+4],0
        MOV     EAX,[.LDivisor+Decimal.Hi]
        ADD     [.RHat],EAX
        JZ      .AdjustLoop96

.AdjustLoopEnd96:

        ;// Finally subtract QHat * divisor from current part of dividend

        MOV     EBX,[.QHat]                   ;// EBX = digit
        MOV     EAX,[.LDivisor+Decimal.Lo]
        MUL     EBX
        MOV     [.A+TAccumulator.L0],EAX
        MOV     ESI,EDX
        MOV     EAX,[.LDivisor+Decimal.Mid]
        MUL     EBX
        ADD     EAX,ESI
        ADC     EDX,0
        MOV     [.A+TAccumulator.L1],EAX
        MOV     ESI,EDX
        MOV     EAX,[.LDivisor+Decimal.Hi]
        MUL     EBX
        ADD     EAX,ESI
        ADC     EDX,0
        MOV     [.A+TAccumulator.L2],EAX

        MOV     EAX,[.A+TAccumulator.L0]
        SUB     DWORD PTR [.LDividend+4*ECX],EAX
        MOV     EAX,[.A+TAccumulator.L1]
        SBB     DWORD PTR [.LDividend+4*ECX+4],EAX
        MOV     EAX,[.A+TAccumulator.L2]
        SBB     DWORD PTR [.LDividend+4*ECX+8],EAX
        SBB     DWORD PTR [.LDividend+4*ECX+12],EDX

        MOV     DWORD PTR [EDI+4*ECX],EBX

        JNC     .EstimateCorrect96

        DEC     DWORD PTR [EDI+4*ECX]
        MOV     EAX,[.LDivisor+Decimal.Lo]
        ADD     DWORD PTR [.LDividend+4*ECX],EAX
        MOV     EAX,[.LDivisor+Decimal.Mid]
        ADC     DWORD PTR [.LDividend+4*ECX+4],EAX
        MOV     EAX,[.LDivisor+Decimal.Hi]
        ADC     DWORD PTR [.LDividend+4*ECX+8],EAX
        ADC     DWORD PTR [.LDividend+4*ECX+12],0

.EstimateCorrect96:

        SUB     ECX,1
        JNC     .MainLoop96

.Exit96:

        MOV     EAX,[.RHat]

        POP     EBX
        POP     EDI
        POP     ESI
end;

;// Finds mantissa modulo 5
;//
;// See Knuth TAOCP, Vol 2, Section 4.3.2, too.
;//
;// Uses modular arithmetic (well, sort of) to convert finding modulus of 4 24 bit words
;// with a few simple additions and multiplications. Uses 24 bit words to avoid values > 32 bit
;// and to allow the multiplication with a magic constant instead of a plain division.
;//
;// Modular arithmetic tells us that:
;//
;//   (a + b) mod n = ((a mod n) + (b mod n)) mod n
;//   (a * b) mod n = ((a mod n) * (b mod n)) mod n
;//
;// Say we have a 96 bit value X. If we separate X into 24 bit words, the bytes are:
;//
;//   X = A2:A1:A0:B2:B1:B0:C2:C1:C0:D2:D1:D0
;//
;// where A2 is the highest and D0 the lowest byte. Let's call the values A, B, C and D.
;// To calculate X mod 5, using the rules above, we get:
;//
;//   X mod 5 = (A * (2^72 mod 5) + B * (2^48 mod 5) + C * (2^24 mod 5) + D) mod 5
;//
;//   2^72 mod 5 = 1
;//   2^48 mod 5 = 1
;//   2^24 mod 5 = 1
;//
;// This finally leads to the formula:
;//
;//   Result := (A + B + C + D) mod 5
;//
;// Approx. 2.5x - 3x as fast as using 3x32 bit words and 7x - 8x as fast as a blind division of
;// all three longwords in the decimal and taking the remainder!

; class function Decimal.Mod96by5(const D: Decimal): Integer;

function Decimal.Mod96by5

        PUSH    EBX

        MOV     EBX,[EAX+8]        ; EBX = A2:A1:A0:B2
        MOV     ECX,[EAX+4]        ; ECX = B1:B0:C2:C1
        MOV     EDX,[EAX]          ; EDX = C0:D2:D1:D0

        MOV     EAX,0
        SHLD    EAX,EBX,24         ; EAX = A2:A1:A0
        SHLD    EBX,ECX,16         ; EBX = B2:B1:B0
        AND     EBX,$00FFFFFF
        SHLD    ECX,EDX,8          ; ECX = C2:C1:C0
        AND     ECX,$00FFFFFF
        AND     EDX,$00FFFFFF      ; EDX = D2:D1:D0

        ADD     EAX,EBX
        ADD     EAX,ECX            ; EAX = A + B + C  (max. $2FFFFFD)
        ADD     EAX,EDX

        MOV     EBX,EAX            ; save EAX

        MOV     ECX,$33333334      ; magic constant = (2^32+4) div 5, i.e. (2^32+n-1) div n
        MUL     ECX                ; EDX = EAX div 5
        LEA     EDX,[EDX+4*EDX]    ; EDX = 5 * (EAX div 5)
        MOV     EAX,EBX
        SUB     EAX,EDX            ; EAX = EAX - 5 * (EAX div 5) = EAX mod 5

        POP     EBX

end;

;class procedure Decimal.InternalDivide(var Result: Decimal; const Left, Right: Decimal);

procedure Decimal.InternalDivide

var LResult,Pointer                     ;// Local copy of Result pointer (which is passed in EAX).
var LDividend,Decimal                   ;// Copy of dividend, required for scaling.
var LDivisor,Decimal                    ;// Normalized divisor (shifted until "top" bit set).
var LQuotient,TAccumulator              ;// Resulting quotient.
var TargetScale,Longword

asm
        PUSH    EBX
        PUSH    ESI

        MOV     [.LResult],EAX

        ;// Need a copy of Left decimal so it can be scaled up. The original should not be touched.

        MOV     ESI,ECX

        MOVZX   EAX,BYTE PTR [EDX+Decimal.Scale]
        MOVZX   ECX,BYTE PTR [ESI+Decimal.Scale]
        SUB     EAX,ECX
        JNS     .TargetScaleNotNegative
        MOV     EAX,0

.TargetScaleNotNegative:

        MOV     [.TargetScale],EAX

        ;// Look for special cases. Thus a 96 bit division can turn into a 64 bit or 32 bit one.

        MOV     EAX,[EDX+Decimal.Lo]
        OR      EAX,[ESI+Decimal.Lo]
        JNE     .NoShiftRight

        MOV     EAX,[EDX+Decimal.Mid]
        OR      EAX,[ESI+Decimal.Mid]
        JE      .ShiftRight64

        ;// Shift dividend and divisor equally right by 32 bit (1 longword).

        MOV     EAX,[EDX+Decimal.Flags]
        MOV     [.LDividend+Decimal.Flags],EAX
        MOV     EAX,[EDX+Decimal.Hi]
        MOV     [.LDividend+Decimal.Mid],EAX
        MOV     EAX,[EDX+Decimal.Mid]
        MOV     [.LDividend+Decimal.Lo],EAX
        MOV     EAX,[ESI+Decimal.Flags]
        MOV     [.LDivisor+Decimal.Flags],EAX
        MOV     EAX,[ESI+Decimal.Hi]
        MOV     [.LDivisor+Decimal.Mid],EAX
        MOV     EAX,[ESI+Decimal.Mid]
        MOV     [.LDivisor+Decimal.Lo],EAX
        MOV     DWORD PTR [.LDividend+Decimal.Hi],0
        MOV     DWORD PTR [.LDivisor+Decimal.Hi],0
        JMP     .ScaleDownDivisor

.ShiftRight64:

        ;// Shift dividend and divisor equally right by 64 bit (2 longwords).

        MOV     EAX,[EDX+Decimal.Hi]
        MOV     [.LDividend+Decimal.Lo],EAX
        MOV     EAX,[ESI+Decimal.Hi]
        MOV     [.LDivisor+Decimal.Lo],EAX
        MOV     EAX,[EDX+Decimal.Flags]
        MOV     [.LDividend+Decimal.Flags],EAX
        MOV     EAX,[ESI+Decimal.Flags]
        MOV     [.LDivisor+Decimal.Flags],EAX
        MOV     DWORD PTR [.LDividend+Decimal.Mid],0
        MOV     DWORD PTR [.LDividend+Decimal.Hi],0
        MOV     DWORD PTR [.LDivisor+Decimal.Mid],0
        MOV     DWORD PTR [.LDivisor+Decimal.Hi],0

.NoShiftRight:

        ;// Cannot shift dividend or divisor right.

        MOV     EAX,[EDX+Decimal.Lo]
        MOV     [.LDividend+Decimal.Lo],EAX
        MOV     EAX,[EDX+Decimal.Mid]
        MOV     [.LDividend+Decimal.Mid],EAX
        MOV     EAX,[EDX+Decimal.Hi]
        MOV     [.LDividend+Decimal.Hi],EAX
        MOV     EAX,[EDX+Decimal.Flags]
        MOV     [.LDividend+Decimal.Flags],EAX

        MOV     ECX,[ESI+Decimal.Flags]
        MOV     [.LDivisor+Decimal.Flags],ECX
        MOV     ECX,[ESI+Decimal.Hi]
        MOV     [.LDivisor+Decimal.Hi],ECX
        MOV     ECX,[ESI+Decimal.Mid]
        MOV     [.LDivisor+Decimal.Mid],ECX
        MOV     ECX,[ESI+Decimal.Lo]
        MOV     [.LDivisor+Decimal.Lo],ECX

        OR      ECX,[.LDivisor+Decimal.Mid]
        OR      ECX,[.LDivisor+Decimal.Hi]
        JNE     .ScaleDownDivisor

        MOV     EAX,detZeroDivide
        CALL    Error

.ScaleDownDivisor:

        ;// If dividend has maximum scale, don't scale down divisor. Must do real division.

        CMP     BYTE PTR [.LDividend+Decimal.Scale],28
        JE      .ScaleLeftLoop

.ScaleDownDivisorLoop:

        ;// Divisor is scaled down, so something like 11/100 does not result in a fraction and then
        ;// in a big number with a high scale, because of the scale-up loop after the division. We
        ;// calculate 11/1 instead (and an accordingly different scale), so no fraction in
        ;// L2:L1:L0 and therefore no scale-up required at the end.

        ;// Can .LDivisor be divided by 2?

        TEST    DWORD PTR [.LDivisor+Decimal.Lo],1
        JNE     .ScaleLeftLoop

        ;// Can .LDivisor be divided by 5?

        LEA     EAX,[.LDivisor]
        CALL    Decimal.Mod96by5
        OR      EAX,EAX
        JNE     .ScaleLeftLoop

        ;// Yes, so it can be divided by 10; scale it down.

        LEA     EAX,[.LDivisor]
        CALL    DecScaleDownRaw
        JMP     .ScaleDownDivisorLoop

.ScaleLeft:

        ;// None of the conditions apply, so scale up.

        LEA    EAX,[.LDividend]
        CALL   DecScaleUpRaw

.ScaleLeftLoop:

        ;// Scale dividend until mantissa of dividend >= mantissa of divisor, if possible. Do
        ;// not omit this step, or otherwise 1 / (1 / 3) returns something like 2.903... instead
        ;// of 3.000...

        ;// Dividend > divisor ?

        MOV    EAX,[.LDividend+Decimal.Lo]
        SUB    EAX,[.LDivisor+Decimal.Lo]
        MOV    EAX,[.LDividend+Decimal.Mid]
        SBB    EAX,[.LDivisor+Decimal.Mid]
        MOV    EAX,[.LDividend+Decimal.Hi]
        SBB    EAX,[.LDivisor+Decimal.Hi]
        JAE    .ScaleLeftEnd

        ;// Enough room to scale up?

        CMP    DWORD PTR [.LDividend+Decimal.Hi],HLWdiv10
        JAE    .ScaleLeftEnd

        ;// Scale not at maximum yet?

        CMP    BYTE PTR [.LDividend+Decimal.Scale],28
        JB     .ScaleLeft

.ScaleLeftEnd:

        LEA     EDX,[.LDividend]

        ;// Result.scale := Left.scale - Right.scale;

        MOV     AL,[.LDividend+Decimal.Scale]
        SUB     AL,[.LDivisor+Decimal.Scale]
        MOV     [.LQuotient+TAccumulator2.Scale],AL

        ;// Result.sign := Left.sign xor Right.sign;

        MOV     AL,[.LDividend+Decimal.Sign]
        XOR     AL,[.LDivisor+Decimal.Sign]
        MOV     [.LQuotient+TAccumulator2.Sign],AL

        ;// Check size of divisor (Right).

        CMP     DWORD PTR [.LDivisor+Decimal.Hi],0
        JNE     .Divisor96
        CMP     DWORD PTR [.LDivisor+Decimal.Mid],0
        JNE     .Divisor64
        MOV     EAX,[.LDivisor+Decimal.Lo]
        OR      EAX,EAX
        JNE     .Divisor32

        ;// Divisor zero.

        MOV     EAX,detZeroDivide
        XOR     EDX,EDX
        JMP     Error

.Divisor32:

        ;// 32 bit divisor.

        LEA     EAX,[.LQuotient]
        MOV     ECX,[.LDivisor+Decimal.Lo]
        CALL    Decimal.Div192by32
        JMP     .ScaleLoop

.Divisor64:

        ;// 64 bit divisor.

        LEA     EAX,[.LQuotient]
        LEA     ECX,[.LDivisor]
        CALL    Decimal.Div192by64
        JMP     .ScaleLoop

.Divisor96:

        ;// 96 bit divisor.

        LEA     EAX,[.LQuotient]
        LEA     ECX,[.LDivisor]
        CALL    Decimal.Div192by96

        ;// Scale up to remove fraction as much as possible.

.ScaleLoop:

        ;// Negative scale? Must scale up until it is 0.

        MOVSX   EDX,BYTE PTR [.LQuotient+TAccumulator2.Scale]
        CMP     EDX,0
        JGE     .ScaleNotNegative
        NEG     EDX
        JMP     .DoScale

.MustScaleDown:

        ;// Scale > 28, so must scale down!

        NEG     EDX
        LEA     EAX,[.LQuotient]
        CALL    AccuScaleDownMax

        CMP     DWORD PTR [.LQuotient+TAccumulator.Underflow],$80000000
        JA      .MustRound
        JB      .MustNotRound
        TEST    DWORD PTR [.LQuotient+TAccumulator.L0],1
        JZ      .MustNotRound

.MustRound:

        ADD     DWORD PTR [.LQuotient+TAccumulator.L0],1
        ADC     DWORD PTR [.LQuotient+TAccumulator.L1],0
        ADC     DWORD PTR [.LQuotient+TAccumulator.L2],0
        ADC     DWORD PTR [.LQuotient+TAccumulator.L3],0
        ADC     DWORD PTR [.LQuotient+TAccumulator.L4],0
        ADC     DWORD PTR [.LQuotient+TAccumulator.L5],0

.MustNotRound:

        ;// Check if scale <= 28 now.

        MOV     EDX,28
        MOVSX   EAX,BYTE PTR [.LQuotient+TAccumulator2.Scale]
        SUB     EDX,EAX
        JS      .MustScaleDown
        JE      .Scaled

.ScaleNotNegative:

        ;// Check if accu can be scaled up.

        CMP     DWORD PTR [.LQuotient+TAccumulator.L6],HLWdiv10
        JAE     .Scaled

        ;// Check if LQuotient.Scale > 28 and scale down if so.

        MOV     EDX,28
        MOVSX   EAX,BYTE PTR [.LQuotient+TAccumulator2.Scale]
        SUB     EDX,EAX
        JS      .MustScaleDown

        ;// if LQuotient.Scale = 28, it can't be scaled up anymore.

        JE      .Scaled

        ;// if fractional part = 0, no need to scale anymore.

        MOV     EAX,[.LQuotient+TAccumulator.L0]
        OR      EAX,[.LQuotient+TAccumulator.L1]
        OR      EAX,[.LQuotient+TAccumulator.L2]
        OR      EAX,[.LQuotient+TAccumulator.L3]
        JE      .Scaled

.DoScale:

        ;// Scale up by EDX

        LEA     EAX,[.LQuotient]
        CALL    AccuScaleUpMax224
        OR      EAX,EAX
        JE      .ScaleLoop

.Scaled:

        ;// Quotient is scaled or can't be scaled anymore.
        ;// If scale < 0, there must be an overflow situation.

        CMP     BYTE PTR [.LQuotient+TAccumulator2.Scale],0
        JGE     .NoOverflow

        MOV     EAX,detOverflow
        CALL    Error

.NoOverflow:

        ;// Check if quotient must be rounded.

        ;// Banker's rounding:
        ;// if > half digit, scale up
        ;// if < half digit, scale down
        ;// if = half digit, scale to nearest even.

        MOV     EAX,[.LQuotient+TAccumulator.L3]
        CMP     EAX,$80000000
        JB      .NoRound
        JA      .DoRound
        TEST    DWORD PTR [.LQuotient+TAccumulator.L4],1
        JE      .NoRound

.DoRound:

        MOV     EAX,[.LQuotient+TAccumulator.L4]
        AND     EAX,[.LQuotient+TAccumulator.L5]
        AND     EAX,[.LQuotient+TAccumulator.L6]
        CMP     EAX,$0FFFFFFFF
        JE      .NoRound
        ADD     DWORD PTR [.LQuotient+TAccumulator.L4],1
        ADC     DWORD PTR [.LQuotient+TAccumulator.L5],0
        ADC     DWORD PTR [.LQuotient+TAccumulator.L6],0

.NoRound:

        ;// Copy top 3 longwords and flags of quotient to Result.

        MOV     EDX,[.LResult]
        MOV     EAX,[.LQuotient+TAccumulator.L4]
        MOV     [EDX+Decimal.Lo],EAX
        MOV     EAX,[.LQuotient+TAccumulator.L5]
        MOV     [EDX+Decimal.Mid],EAX
        MOV     EAX,[.LQuotient+TAccumulator.L6]
        MOV     [EDX+Decimal.Hi],EAX
        MOV     EAX,[.LQuotient+TAccumulator.Flags]
        MOV     AX,0
        MOV     [EDX+Decimal.Flags],EAX

        ;// Try to eliminate trailing zero digits, if scale > target scale

        ;// This is not without cost, but required to get correct result, until I find
        ;// a better way to achieve this (e.g. not scaling up further than necessary).

        MOV     EAX,[EDX+Decimal.Lo]
        OR      EAX,[EDX+Decimal.Mid]
        OR      EAX,[EDX+Decimal.Hi]
        JE      .Exit
        MOV     EAX,EDX
        MOV     EDX,[.TargetScale]
        CALL    DecScaleDownTo

        ;// Think about this:
        ;//
        ;// It is possible to find the maximum number that an accu or decimal can be scaled up
        ;// by counting the leading zero bits (up to 95). This is in fact the 2log. Multiplying
        ;// that number by 10log(2) = 0.30103 then gives a 10log.
        ;// This can be approximated by e.g. 1233/4096 (* 1233, shr 12) = 0.301025, or
        ;// 5050445/1677726 (* 5050445, shr 24) = 0.301029, although it can be one too low, so
        ;// after that, a test against HLWdiv10 is required and another scale up may be necessary.
        ;//
        ;// The lowest bit set of an accu or decimal is significant in two ways:
        ;// 1. the number of trailing zeroes determines how far the decimal or accu can be
        ;//    scaled down maximally without losing information, since 10 = 5 * 2. Mod96by5 can
        ;//    help to determine if it can also be divided by 5.
        ;// 2. the lowest bit (of the fractional part) determines, how many times an accu must be
        ;//    scaled up to clear the fractional part.
        ;//
        ;// Not sure how this can be used in the scaling part above.

.Exit:

        POP     ESI
        POP     EBX
end;


; class procedure Decimal.InternalFloor(var Result: Decimal; const Source: Decimal);

procedure Decimal.InternalFloor

asm
        PUSH    EDI

        MOV     EDI,EAX
        CALL    Decimal.InternalTruncate
        CMP     BYTE PTR [EDI+Decimal.Sign],0
        JE      .Exit1
        ADD     DWORD PTR [EDI+Decimal.Lo],1
        ADC     DWORD PTR [EDI+Decimal.Mid],0
        ADC     DWORD PTR [EDI+Decimal.Hi],0

.Exit1:

        POP     EDI
end;

;// Basecase multiplication
;//
;// Karatsuba, Divide and Conquer, etc. are probably only useful for larger sizes.

; class procedure Decimal.InternalMultiply(var Result: Decimal;
;   const Left, Right: Decimal);

procedure Decimal.InternalMultiply

var LResult,Pointer
var A,TAccumulator

asm
        PUSH    ESI
        PUSH    EDI
        PUSH    EBX

        MOV     [.LResult],EAX

        MOV     ESI,EDX
        MOV     EDI,ECX
        MOV     ECX,EAX
        XOR     EAX,EAX
        MOV     [.A+TAccumulator.L0],EAX
        MOV     [.A+TAccumulator.L1],EAX
        MOV     [.A+TAccumulator.L2],EAX
        MOV     [.A+TAccumulator.L3],EAX
        MOV     [.A+TAccumulator.L4],EAX
        MOV     [.A+TAccumulator.L5],EAX
        MOV     [.A+TAccumulator.Underflow],EAX
        MOV     [.A+TAccumulator.Flags],EAX

        ;// Add exponents

        MOV     AL,[ESI+Decimal.Scale]
        ADD     AL,[EDI+Decimal.Scale]
        MOV     [.A+TAccumulator2.Scale],AL

        ;// Set sign

        MOV     AL,[ESI+Decimal.Sign]
        XOR     AL,[EDI+Decimal.Sign]
        MOV     [.A+TAccumulator2.Sign],AL

;
;                         [ RH ][ RM ][ RL ]
;                         [ LH ][ LM ][ LL ]
;                         ------------------ x
;                               [   LL*RL  ]
;                         [   LL*RM  ]   |
;                   [   LL*RH  ]   |     |
;                         [   LM*RL  ]   |
;                   [   LM*RM  ]   |     |
;             [   LM*RH  ]   |     |     |
;                   [   LH*RL  ]   |     |
;             [   LH*RM  ]   |     |     |
;       [   LH*RH  ]         |     |     |
;       ---v-----v-----v-----v-----v-----v-- +
;       [ A5 ][ A4 ][ A3 ][ A2 ][ A1 ][ A0 ]
;

        MOV     EBX,[ESI+Decimal.Lo]         ;// ESI:ECX:EBX is mantissa of Left.
        OR      EBX,EBX
        JE      .LeftMid

        ;// Left.Lo * Right.Lo

        MOV     EAX,[EDI+Decimal.Lo]
        MUL     EBX
        MOV     [.A+TAccumulator.L0],EAX
        MOV     [.A+TAccumulator.L1],EDX

        ;// Left.Lo * Right.Mid

        MOV     EAX,[EDI+Decimal.Mid]
        MUL     EBX
        ADD     [.A+TAccumulator.L1],EAX
        ADC     [.A+TAccumulator.L2],EDX

        ;// Left.Lo * Right.Hi

        MOV     EAX,[EDI+Decimal.Hi]
        MUL     EBX
        ADD     [.A+TAccumulator.L2],EAX
        ADC     [.A+TAccumulator.L3],EDX

.LeftMid:

        MOV     EBX,[ESI+Decimal.Mid]
        OR      EBX,EBX
        JE      .LeftHi

        ;// Left.Mid * Right.Lo

        MOV     EAX,[EDI+Decimal.Lo]
        MUL     EBX
        ADD     [.A+TAccumulator.L1],EAX
        ADC     [.A+TAccumulator.L2],EDX
        ADC     DWORD PTR [.A+TAccumulator.L3],0

        ;// Left.Mid * Right.Mid

        MOV     EAX,[EDI+Decimal.Mid]
        MUL     EBX
        ADD     [.A+TAccumulator.L2],EAX
        ADC     [.A+TAccumulator.L3],EDX
        ADC     DWORD PTR [.A+TAccumulator.L4],0

        ;// Left.Mid * Right.Hi

        MOV     EAX,[EDI+Decimal.Hi]
        MUL     EBX
        ADD     [.A+TAccumulator.L3],EAX
        ADC     [.A+TAccumulator.L4],EDX
        ADC     DWORD PTR [.A+TAccumulator.L5],0

.LeftHi:

        MOV     EBX,[ESI+Decimal.Hi]
        OR      EBX,EBX
        JE      .CheckTop

        ;// Left.Hi * Right.Lo

        MOV     EAX,[EDI+Decimal.Lo]
        MUL     EBX
        ADD     [.A+TAccumulator.L2],EAX
        ADC     [.A+TAccumulator.L3],EDX
        ADC     DWORD PTR [.A+TAccumulator.L4],0

        ;// Left.Hi * Right.Mid

        MOV     EAX,[EDI+Decimal.Mid]
        MUL     EBX
        ADD     [.A+TAccumulator.L3],EAX
        ADC     [.A+TAccumulator.L4],EDX
        ADC     DWORD PTR [.A+TAccumulator.L5],0

        ;// Left.Hi * Right.Hi

        MOV     EAX,[EDI+Decimal.Hi]
        MUL     EBX
        ADD     [.A+TAccumulator.L4],EAX
        ADC     [.A+TAccumulator.L5],EDX

.CheckTop:

        MOV     AL,[.A+TAccumulator2.Scale]
        SUB     AL,28
        JLE     .CheckL5
        MOVZX   EDX,AL
        LEA     EAX,[.A]
        CALL    AccuScaleDownMax
        JMP     .CheckTop

.CheckL5:

        CMP     DWORD PTR [.A+TAccumulator.L5],0
        JE      .CheckL4
        CMP     BYTE PTR [.A+TAccumulator2.Scale],0
        JE      .Overflow
        LEA     EAX,[.A]
        MOV     EDX,9
        CALL    AccuScaleDownMax
        JMP     .CheckL5

.CheckL4:

        CMP     DWORD PTR [.A+TAccumulator.L4],0
        JE      .CheckL3
        CMP     BYTE PTR [.A+TAccumulator2.Scale],0
        JE      .Overflow
        LEA     EAX,[.A]
        MOV     EDX,9
        CALL    AccuScaleDownMax
        JMP     .CheckL4

.CheckL3:

        ;// Loop fully unrolled, for speed:

        MOV     ECX,[.A+TAccumulator.L3]
        CMP     ECX,0
        JE      .Exit
        CMP     BYTE PTR [.A+TAccumulator2.Scale],0
        JE      .Overflow
        LEA     EAX,[.A]
        CMP     ECX,10
        JNB     .Not10
        CALL    AccuScaleDownRaw128
        JMP     .Exit

.Not10:

        CMP     ECX,100
        JNB     .Not100
        MOV     EDX,2
        CALL    AccuScaleDownMax128
        JMP     .Exit

.Not100:

        CMP     ECX,1000
        JNB     .Not1000
        MOV     EDX,3
        CALL    AccuScaleDownMax128
        JMP     .Exit

.Not1000:

        CMP     ECX,10*1000
        JNB     .Not10000
        MOV     EDX,4
        CALL    AccuScaleDownMax128
        JMP     .Exit

.Not10000:

        CMP     ECX,100*1000
        JNB     .Not100000
        MOV     EDX,5
        CALL    AccuScaleDownMax128
        JMP     .Exit

.Not100000:

        CMP     ECX,1000*1000
        JNB     .Not1000000
        MOV     EDX,6
        CALL    AccuScaleDownMax128
        JMP     .Exit

.Not1000000:

        CMP     ECX,10*1000*1000
        JNB     .Not10000000
        MOV     EDX,7
        CALL    AccuScaleDownMax128
        JMP     .Exit

.Not10000000:

        CMP     ECX,100*1000*1000
        JNB     .Not100000000
        MOV     EDX,8
        CALL    AccuScaleDownMax128
        JMP     .Exit

.Not100000000:

        MOV     EDX,9
        CALL    AccuScaleDownMax128
        JMP     .CheckL3

.Exit:

        CMP     DWORD PTR [.A+TAccumulator.Underflow],$80000000
        JB      .NoRound
        JA      .DoRound
        TEST    DWORD PTR [.A+TAccumulator.L0],1
        JZ      .NoRound

.DoRound:

        ADD     DWORD PTR [.A+TAccumulator.L0],1
        ADC     DWORD PTR [.A+TAccumulator.L1],0
        ADC     DWORD PTR [.A+TAccumulator.L2],0
        ADC     DWORD PTR [.A+TAccumulator.L3],0

.NoRound:

        CMP     DWORD PTR [.A+TAccumulator.L3],0
        JZ      .NoOverflow

        CMP     BYTE PTR [.A+TAccumulator2.Scale],0
        JLE     .Overflow

        LEA     EAX,[.A]
        CALL    AccuScaleDownRaw128
        JMP     .NoOverflow

.Overflow:

        MOV     EAX,detOverflow
        XOR     EDX,EDX
        JMP     Error

.NoOverflow:

        CMP     BYTE PTR [.A+TAccumulator2.Scale],0
        JL      .Overflow

        MOV     EDI,[.LResult]
        MOV     EAX,[.A+TAccumulator.L0]
        MOV     [EDI+Decimal.Lo],EAX
        MOV     EAX,[.A+TAccumulator.L1]
        MOV     [EDI+Decimal.Mid],EAX
        MOV     EAX,[.A+TAccumulator.L2]
        MOV     [EDI+Decimal.Hi],EAX
        MOV     EAX,[.A+TAccumulator.Flags]
        MOV     [EDI+Decimal.Flags],EAX

        POP     EBX
        POP     EDI
        POP     ESI
end;

; class procedure Decimal.InternalRound(var Result: Decimal; const Source: Decimal);

procedure Decimal.InternalRound

asm
        PUSH    EDI

        MOV     EDI,EAX
        MOV     EAX,[EDX+Decimal.Lo]
        MOV     [EDI+Decimal.Lo],EAX
        MOV     EAX,[EDX+Decimal.Mid]
        MOV     [EDI+Decimal.Mid],EAX
        MOV     EAX,[EDX+Decimal.Hi]
        MOV     [EDI+Decimal.Hi],EAX
        MOV     EAX,[EDX+Decimal.Flags]
        MOV     [EDI+Decimal.Flags],EAX

.Loop:

        CMP     BYTE PTR [EDI+Decimal.Scale],0
        JLE     .EndLoop
        MOV     EAX,EDI
        MOVZX   EDX,BYTE PTR [EDI+Decimal.Scale]
        CALL    DecScaleDownMaxRaw
        JMP     .Loop

.EndLoop:

        POP     EDI
end;

; class procedure Decimal.InternalToDouble(var Result: Double; const Source: Decimal);

procedure Decimal.InternalToDouble

var     A,TAccumulator
var     LResult,Pointer

asm
        PUSH    EDI
        PUSH    ESI
        PUSH    EBX
        MOV     [.LResult],EAX

        XOR     EAX,EAX
        MOV     ESI,EDX
        XOR     EAX,EAX
        MOV     [.A+TAccumulator.L0],EAX
        MOV     [.A+TAccumulator.L1],EAX
        MOV     [.A+TAccumulator.L2],EAX
        MOV     EAX,[ESI+Decimal.Lo]
        MOV     [.A+TAccumulator.L3],EAX
        MOV     EAX,[ESI+Decimal.Mid]
        MOV     [.A+TAccumulator.L4],EAX
        MOV     EAX,[ESI+Decimal.Hi]
        MOV     [.A+TAccumulator.L5],EAX
        MOV     EAX,[ESI+Decimal.Flags]
        MOV     [.A+TAccumulator.Flags],EAX

.ScaleLoop:

        CMP     BYTE PTR [.A+TAccumulator2.Scale],0
        JE      .Scaled
        LEA     EAX,[.A]
        CALL    AccuScaleDownRaw
        JMP     .ScaleLoop

.Scaled:

        MOV     EDX,5

.FindLoop:

        CMP     DWORD PTR [.A+4*EDX],0
        JNE     .FoundTop
        DEC     EDX
        JNS     .FindLoop
        JMP     .IsZero

.FoundTop:

        MOV     ESI,DWORD PTR [.A+4*EDX]  ;// ESI:EDI:EBX contain significant bits
        CMP     EDX,0
        JE      .NoLower0
        MOV     EDI,DWORD PTR [.A+4*EDX-4]
        CMP     EDX,1
        JE      .NoLower1
        MOV     EBX,DWORD PTR [.A+4*EDX-8]
        JMP     .FindBit

.NoLower0:

        XOR     EDI,EDI

.NoLower1:

        XOR     EBX,EBX

.FindBit:

        BSR     EAX,ESI
        JZ      .IsZero
        SHL     EDX,5
        OR      EDX,EAX                  ;// EDX = exponent
        ADD     EDX,DoubleBias-96
        MOV     ECX,EAX
        NEG     ECX
        JZ      .ZeroShift
        SHLD    ESI,EDI,CL
        SHLD    EDI,EBX,CL
        JMP     .Shifted

.ZeroShift:

        MOV     ESI,EDI
        MOV     EDI,EBX

.Shifted:

        MOV     EAX,EDI                  ;// ESI:EDI = mantissa shifted fully left
        TEST    EAX,$0800
        JE      .NoRound
        ADD     EAX,$1000
        ADC     ESI,0
        ADC     EDX,0

.NoRound:

        SHRD    EAX,ESI,12               ;// ESI:EAX = mantissa
        SHR     ESI,12
        AND     EDX,DoubleEMask
        SHL     EDX,52-32
        OR      EDX,ESI
        MOV     ESI,[.A+TAccumulator.Flags]
        AND     ESI,$80000000
        OR      EDX,ESI
        JMP     .Finish

.IsZero:

        XOR     EAX,EAX
        XOR     EDX,EDX

.Finish:

        MOV     EDI,[.LResult]
        MOV     [EDI],EAX
        MOV     [EDI+4],EDX

        POP     EBX
        POP     ESI
        POP     EDI
end;

;class procedure Decimal.InternalToExtended(var Result: Extended;
;  const Source: Decimal);

procedure Decimal.InternalToExtended

var     A,TAccumulator

asm
        PUSH    EDI
        PUSH    ESI
        PUSH    EBX
        PUSH    EAX

        XOR     EAX,EAX
        MOV     ESI,EDX
        XOR     EAX,EAX
        MOV     [.A+TAccumulator.L0],EAX
        MOV     [.A+TAccumulator.L1],EAX
        MOV     [.A+TAccumulator.L2],EAX
        MOV     EAX,[ESI+Decimal.Lo]
        MOV     [.A+TAccumulator.L3],EAX
        MOV     EAX,[ESI+Decimal.Mid]
        MOV     [.A+TAccumulator.L4],EAX
        MOV     EAX,[ESI+Decimal.Hi]
        MOV     [.A+TAccumulator.L5],EAX
        MOV     EAX,[ESI+Decimal.Flags]
        MOV     [.A+TAccumulator.Flags],EAX
.ScaleLoop:
        CMP     BYTE PTR [.A+TAccumulator2.Scale],0
        JE      .Scaled
        LEA     EAX,[.A]
        CALL    AccuScaleDownRaw
        JMP     .ScaleLoop
.Scaled:
        MOV     EDX,5
.FindLoop:
        CMP     DWORD PTR [.A+4*EDX],0
        JNE     .FoundTop
        DEC     EDX
        JNS     .FindLoop
        JMP     .IsZero
.FoundTop:
        MOV     ESI,DWORD PTR [.A+4*EDX]  ;// ESI:EDI:EBX contain significant bits
        CMP     EDX,0
        JE      .NoLower0
        MOV     EDI,DWORD PTR [.A+4*EDX-4]
        CMP     EDX,1
        JE      .NoLower1
        MOV     EBX,DWORD PTR [.A+4*EDX-8]
        JMP     .FindBit
.NoLower0:
        XOR     EDI,EDI
.NoLower1:
        XOR     EBX,EBX
.FindBit:
        BSR     EAX,ESI
        JZ      .IsZero
        SHL     EDX,5
        OR      EDX,EAX
        ADD     EDX,ExtendedBias-96      ;// EDX = exponent

        MOV     ECX,31
        SUB     ECX,EAX
        JZ      .Shifted
        SHLD    ESI,EDI,CL
        SHLD    EDI,EBX,CL
        SHL     EBX,CL
        JMP     .Shifted
.Shifted:
        MOV     EAX,EDI
        CMP     EBX,$80000000
        JB      .NoRound
        ADD     EAX,1
        ADC     ESI,0
        JNC     .NoRound
        ADC     EDX,0
        STC
        RCR     ESI,1
        RCR     EAX,1
.NoRound:
        XOR     ECX,ECX
        MOV     CH,[.A+TAccumulator2.Sign]
        AND     EDX,$7FFF
        OR      ECX,EDX
        MOV     EDX,ESI
        JMP     .Finish
.IsZero:
        XOR     EAX,EAX
        XOR     EDX,EDX
        XOR     ECX,ECX
.Finish:
        POP     EDI
        MOV     [EDI],EAX
        MOV     [EDI+4],EDX
        MOV     [EDI+8],CX

        POP     EBX
        POP     ESI
        POP     EDI
end;

; class procedure Decimal.InternalToSingle(var Result: Single; const Source: Decimal);

procedure Decimal.InternalToSingle

var     A,TAccumulator

asm
        PUSH    EDI
        PUSH    ESI
        PUSH    EAX

        XOR     EAX,EAX
        MOV     ESI,EDX
        MOV     [.A+TAccumulator.L0],EAX
        MOV     [.A+TAccumulator.L1],EAX
        MOV     [.A+TAccumulator.L2],EAX
        MOV     EAX,[EDX+Decimal.Lo]
        MOV     [.A+TAccumulator.L3],EAX
        MOV     EAX,[EDX+Decimal.Mid]
        MOV     [.A+TAccumulator.L4],EAX
        MOV     EAX,[EDX+Decimal.Hi]
        MOV     [.A+TAccumulator.L5],EAX
        MOV     EAX,[EDX+Decimal.Flags]
        MOV     [.A+TAccumulator.Flags],EAX

.ScaleLoop:

        CMP     BYTE PTR [.A+TAccumulator2.Scale],0
        JE      .Scaled
        LEA     EAX,[.A]
        CALL    AccuScaleDownRaw
        JMP     .ScaleLoop

.Scaled:

        MOV     EDX,5

.FindLoop:

        CMP     DWORD PTR [.A+4*EDX],0
        JNE     .FoundTop
        DEC     EDX
        JNS     .FindLoop
        JMP     .IsZero

.FoundTop:

        MOV     ESI,DWORD PTR [.A+4*EDX]
        CMP     EDX,0
        JE      .NoLower
        MOV     EDI,DWORD PTR [.A+4*EDX-4]
        JMP     .FindBit

.NoLower:

        XOR     EDI,EDI

.FindBit:

        BSR     EAX,ESI
        JZ      .IsZero
        SHL     EDX,5
        OR      EDX,EAX                  ;// EDX = exponent
        ADD     EDX,SingleBias-96
        MOV     ECX,EAX
        NEG     ECX
        JZ      .ZeroShift
        SHLD    ESI,EDI,CL
        JMP     .Shifted

.ZeroShift:

        MOV     ESI,EDI

.Shifted:

        MOV     EAX,ESI                  ;// ESI = mantissa
        TEST    EAX,$100
        JE      .NoRound
        ADD     EAX,$200
        ADC     EDX,0

.NoRound:

        SHR     EAX,9
        AND     EDX,$0FF
        SHL     EDX,23
        OR      EAX,EDX
        MOV     EDX,[.A+TAccumulator.Flags]
        AND     EDX,$80000000
        OR      EAX,EDX
        JMP     .Finish

.IsZero:

        XOR     EAX,EAX

.Finish:

        POP     EDI
        MOV     [EDI],EAX

        POP     ESI
        POP     EDI
end;

; class procedure Decimal.InternalTruncate(var Result: Decimal;
;   const Source: Decimal);

procedure Decimal.InternalTruncate

asm
        PUSH    EDI
        PUSH    ESI

        MOV     ESI,EDX
        MOV     EDI,EAX
        MOV     EAX,[EDX+Decimal.Lo]
        MOV     [EDI+Decimal.Lo],EAX
        MOV     EAX,[EDX+Decimal.Mid]
        MOV     [EDI+Decimal.Mid],EAX
        MOV     EAX,[EDX+Decimal.Hi]
        MOV     [EDI+Decimal.Hi],EAX
        MOV     EAX,[EDX+Decimal.Flags]
        MOV     [EDI+Decimal.Flags],EAX

.ScaleDownLoop:

        MOVZX   EDX,BYTE PTR [EDI+Decimal.Scale]
        OR      EDX,EDX
        JLE     .EndLoop
        CMP     EDX,9
        JLE     .ScaleOK
        MOV     EDX,9

.ScaleOK:

        SUB     [EDI+Decimal.Scale],DL
        MOV     ECX,DWORD PTR [PowersOfTen+4*EDX]
        XOR     EDX,EDX
        MOV     EAX,[EDI+Decimal.Hi]
        DIV     ECX
        MOV     [EDI+Decimal.Hi],EAX
        MOV     EAX,[EDI+Decimal.Mid]
        DIV     ECX
        MOV     [EDI+Decimal.Mid],EAX
        MOV     EAX,[EDI+Decimal.Lo]
        DIV     ECX
        MOV     [EDI+Decimal.Lo],EAX
        JMP     .ScaleDownLoop

.EndLoop:

        POP     ESI
        POP     EDI
end;

; function Decimal.ToStringWithSettings(const Settings: TFormatSettings): string;

function Decimal.ToStringWithSettings

var     Buffer,Char_size*51
var     D,Decimal
var     DecimalSeparator,Char

asm
        PUSH    EDI
        PUSH    ESI
        PUSH    EBX
        PUSH    ECX             ;// Result string

        MOV     EBX,EAX
        MOV     ESI,EAX
        MOV     AX,[EDX+TFormatSettings.DecimalSeparator]
        MOV     [.DecimalSeparator],AX
        MOV     EAX,[ESI+Decimal.Lo]
        MOV     [.D+Decimal.Lo],EAX
        MOV     EAX,[ESI+Decimal.Mid]
        MOV     [.D+Decimal.Mid],EAX
        MOV     EAX,[ESI+Decimal.Hi]
        MOV     [.D+Decimal.Hi],EAX
        MOV     EAX,[ESI+Decimal.Flags]
        MOV     [.D+Decimal.Flags],EAX

        LEA     ESI,[.Buffer+2*50]
        XOR     EDI,EDI

        MOV     WORD PTR [ESI],0
        SUB     ESI,2
        INC     EDI

.FractionLoop:

        CMP     BYTE PTR [.D+Decimal.Scale],0
        JLE     .EndFractionLoop
        LEA     EAX,[.D]
        CALL    DecScaleDownRaw
        ADD     AL,'0'
        MOV     AH,0
        MOV     [ESI],AX
        SUB     ESI,2
        INC     EDI
        JMP     .FractionLoop

.EndFractionLoop:

        CMP     BYTE PTR [EBX+Decimal.Scale],0
        JE      .IntegerLoop
        MOV     AX,[.DecimalSeparator]
        MOV     [ESI],AX
        SUB     ESI,2
        INC     EDI

.IntegerLoop:

        LEA     EAX,[.D]
        CALL    DecScaleDownRaw
        ADD     AL,'0'
        MOV     AH,0
        MOV     [ESI],AX
        SUB     ESI,2
        INC     EDI
        MOV     EAX,[.D+Decimal.Lo]
        OR      EAX,[.D+Decimal.Mid]
        OR      EAX,[.D+Decimal.Hi]
        JNE     .IntegerLoop
        CMP     BYTE PTR [.D+Decimal.Sign],0
        JE      .Exit
        MOV     WORD PTR [ESI],'-'
        SUB     ESI,2
        INC     EDI

.Exit:

        POP     EAX             ;// Result string
        MOV     EDX,ESI
        ADD     EDX,2           ;// PChar to string in buffer
        MOV     ECX,EDI         ;// Length of string
        CALL    __UStrFromPWCharLen

        POP     EBX
        POP     ESI
        POP     EDI
end;


; function Decimal.ToStringBlank: string;

function Decimal.ToStringBlank

var     Buffer,Char_size*51
var     D,Decimal

asm
        PUSH    EDI
        PUSH    ESI
        PUSH    EBX
        PUSH    EDX

        MOV     EBX,EAX
        MOV     EAX,[EBX+Decimal.Lo]
        MOV     [.D+Decimal.Lo],EAX
        MOV     EAX,[EBX+Decimal.Mid]
        MOV     [.D+Decimal.Mid],EAX
        MOV     EAX,[EBX+Decimal.Hi]
        MOV     [.D+Decimal.Hi],EAX
        MOV     EAX,[EBX+Decimal.Flags]
        MOV     [.D+Decimal.Flags],EAX
        LEA     ESI,[.Buffer+50*2]
        XOR     EDI,EDI

        MOV     WORD PTR [ESI],0
        SUB     ESI,2

.FractionLoop:

        CMP     BYTE PTR [.D+Decimal.Scale],0
        JLE     .EndFractionLoop
        LEA     EAX,[.D]
        CALL    DecScaleDownRaw
        ADD     AL,'0'
        MOV     AH,0
        MOV     [ESI],AX
        SUB     ESI,2
        INC     EDI
        JMP     .FractionLoop

.EndFractionLoop:

        CMP     BYTE PTR [EBX+Decimal.Scale],0
        JE      .IntegerLoop
        MOV     AX,[DecimalSeparator]
        MOV     [ESI],AX
        SUB     ESI,2
        INC     EDI

.IntegerLoop:

        LEA     EAX,[.D]
        CALL    DecScaleDownRaw
        ADD     AL,'0'
        MOV     AH,0
        MOV     [ESI],AX
        SUB     ESI,2
        INC     EDI
        MOV     EAX,[.D+Decimal.Lo]
        OR      EAX,[.D+Decimal.Mid]
        OR      EAX,[.D+Decimal.Hi]
        JNE     .IntegerLoop
        CMP     BYTE PTR [.D+Decimal.Sign],0
        JE      .Exit
        MOV     WORD PTR [ESI],'-'
        SUB     ESI,2
        INC     EDI

.Exit:

        POP     EAX             ;// Result string
        MOV     EDX,ESI
        ADD     EDX,2           ;// PChar to string in buffer
        MOV     ECX,EDI         ;// Length of string
        CALL    __UStrFromPWCharLen

        POP     EBX
        POP     ESI
        POP     EDI
end;


;// Tries to parse S as Decimal. Returns True if successful.
; class function Decimal.TryParse(const S: string;
;   const Settings: TFormatSettings; out Res: Decimal): Boolean;

function Decimal.TryParseWithSettings

var     InExponent,Boolean
var     IsNegative,Boolean
var     IsNegativeExponent,Boolean
var     Exponent,Integer
var     NumDecimals,Integer
var     DecimalPointPos,Pointer
var     P,Pointer
var     DecimalSep,Char
var     ThousandSep,Char
var     Result,Boolean

asm
        PUSH    EDI
        PUSH    ESI
        PUSH    EBX
        MOV     BYTE PTR [.Result],0
        MOV     ESI,EAX
        MOV     AX,[EDX+TFormatSettings.DecimalSeparator]
        MOV     [.DecimalSep],AX
        MOV     AX,[EDX+TFormatSettings.ThousandSeparator]
        MOV     [.ThousandSep],AX
        MOV     EBX,ECX
        MOV     EAX,ESI
        LEA     EDX,[.P]
        CALL    Trim
        MOV     EAX,[.P]
        OR      EAX,EAX
        JE      .Exit
        MOV     ESI,EAX
        XOR     EAX,EAX
        MOV     [.InExponent],AL
        MOV     [.IsNegative],AL
        MOV     [.IsNegativeExponent],AL
        MOV     [.Exponent],EAX
        MOV     [.NumDecimals],EAX
        MOV     [.DecimalPointPos],EAX
        MOV     EDI,EBX
        MOV     [EBX+Decimal.Lo],EAX
        MOV     [EBX+Decimal.Mid],EAX
        MOV     [EBX+Decimal.Hi],EAX
        MOV     [EBX+Decimal.Flags],EAX

.MainLoop:

        MOV     AX,[ESI]
        CMP     AX,0
        JE      .EndMainLoop
        CMP     AX,'+'
        JE      .Plus
        CMP     AX,'-'
        JE      .Minus
        CMP     AX,'.'
        JE      .DecimalPoint
        CMP     AX,[.DecimalSep]
        JE      .DecimalPoint
        CMP     AX,[.ThousandSep]
        JE      .Thousands
        CMP     AX,'E'
        JE      .Scientific
        CMP     AX,'e'
        JE      .Scientific
        CMP     AX,'0'
        JL      .EndCase
        CMP     AX,'9'
        JLE     .Digit

.EndCase:

        ADD     ESI,2
        JMP     .MainLoop

.Plus:

        CMP     BYTE PTR [.InExponent],0
        JE      .Pls00
        CMP     ESI,[.P]
        JLE     .Exit
        ADD     ESI,2
        JMP     .MainLoop

.Pls00:

        CMP     ESI,[.P]
        JNE     .Exit
        ADD     ESI,2
        JMP     .MainLoop

.Minus:

        CMP     BYTE PTR [.InExponent],0
        JE      .Min00
        CMP     ESI,[.P]
        JLE     .Exit
        MOV     BYTE PTR [.IsNegativeExponent],1
        ADD     ESI,2
        JMP     .MainLoop

.Min00:

        CMP     ESI,[.P]
        JNE     .Exit
        MOV     BYTE PTR [.IsNegative],1
        ADD     ESI,2
        JMP     .MainLoop

.DecimalPoint:

        CMP     BYTE PTR [.DecimalPointPos],0
        JNE     .Exit
        MOV     [.DecimalPointPos],ESI

.Thousands:

        ADD     ESI,2
        JMP     .MainLoop

.Scientific:

        MOV     BYTE PTR [.InExponent],1
        CMP     DWORD PTR [.DecimalPointPos],0
        JE      .EndCase
        MOV     EAX,ESI
        SUB     EAX,[.DecimalPointPos]
        SAR     EAX,1
        DEC     EAX
        MOV     [.NumDecimals],EAX
        MOV     DWORD PTR [.DecimalPointPos],0
        ADD     ESI,2
        JMP     .MainLoop

.Digit:

        SUB     AX,'0'
        MOVZX   EBX,AX
        CMP     BYTE PTR [.InExponent],0
        JE      .Dig00
        MOV     ECX,10
        MOV     EAX,[.Exponent]
        MUL     ECX
        ADD     EAX,EBX
        MOV     [.Exponent],EAX
        ADD     ESI,2
        JMP     .MainLoop

.Dig00:

        CMP     DWORD PTR [EDI+Decimal.Hi],HLWdiv10
        JA      .ExcessDigits
        JB      .NoExcess
        CMP     DWORD PTR [EDI+Decimal.Mid],HUI64div10
        JA      .ExcessDigits

.NoExcess:

        MOV     EDX,EBX
        MOV     EAX,EDI
        CALL    DecScaleUpAndAdd
        ADD     ESI,2
        JMP     .MainLoop

.ExcessDigits:

        CMP     EBX,5
        JA      .DoRound
        JB      .NoRound
        TEST    DWORD PTR [EDI+Decimal.Lo],1
        JZ      .NoRound

.DoRound:

        ADD     DWORD PTR [EDI+Decimal.Lo],1
        ADC     DWORD PTR [EDI+Decimal.Mid],0
        ADC     DWORD PTR [EDI+Decimal.Hi],0

.NoRound:

        JMP     .EndMainLoop

.OverflowError:

        MOV     EAX,detOverflow
        JMP     Error

.UnderflowError:

        MOV     EAX,detUnderflow
        JMP     Error

.EndMainLoop:

        MOV     EAX,[.DecimalPointPos]
        OR      EAX,EAX
        JE      .SkipDecPoint
        MOV     EDX,ESI
        SUB     EDX,EAX
        SAR     EDX,1
        DEC     EDX
        MOV     [.NumDecimals],EDX

.SkipDecPoint:

        CMP     BYTE PTR [.IsNegativeExponent],1
        JNE     .SkipNegExp
        NEG     DWORD PTR [.Exponent]

.SkipNegExp:

        MOV     EAX,[.NumDecimals]
        SUB     EAX,[.Exponent]
        MOV     [.Exponent],EAX
        CMP     DWORD PTR [.Exponent],-28
        JLE     .OverflowError

.ScaleDownLoop:

        CMP     DWORD PTR [.Exponent],28
        JLE     .ScaleUpLoop
        MOV     EAX,EDI
        CALL    DecScaleDownRaw
        DEC     DWORD PTR [.Exponent]
        JNE     .ScaleDownLoop

.ScaleUpLoop:

        CMP     DWORD PTR [.Exponent],0
        JGE     .CheckOverflow
        MOV     EAX,EDI
        CALL    DecScaleUpRaw
        INC     DWORD PTR [.Exponent]
        JLE     .ScaleUpLoop

.CheckOverflow:

        CMP     DWORD PTR [.Exponent],0
        JL      .OverflowError

.CheckNegative:

        CMP     BYTE PTR [.IsNegative],1
        JNE     .SkipNegative
        OR      BYTE PTR [EDI+Decimal.Sign],$80

.SkipNegative:

        MOV     EAX,[.Exponent]
        MOV     BYTE PTR [EDI+Decimal.Scale],AL
        MOV     BYTE PTR [.Result],1

.Exit:

        MOVZX   EAX,BYTE PTR [.Result]

        POP     EBX
        POP     ESI
        POP     EDI
end;

