﻿{                                                                           }
{ File:       Decimals_nasm.pas                                             }
{ Function:   Decimal data type, an emulation of the type of the same name  }
{             in .NET. It is a floating decimal point binary mantissa data  }
{             type completely calculated in software, so it is slow, but    }
{             well suited for calculations that require accuracy,           }
{             especially financial ones.                                    }
{ Language:   Delphi XE                                                     }
{ Author:     Rudy Velthuis                                                 }
{ Copyright:  (c) 2010 Rudy Velthuis                                        }
{ Version:    2.5a (using external assembler)                               }
{ References: Donald Knuth, The Art of Computer Programming, Vol. 2:        }
{             Seminumerical Algorithms                                      }
{ Notes:      This file is a copy of Decimals.pas, but it doesn't use       }
{             the Delphi built-in assembler. It relies on decimals.obj,     }
{             which is generated using the NASM assembler.                  }
{                                                                           }
{ Disclaimer: Redistribution and use in source and binary forms, with or    }
{             without modification, are permitted provided that the         }
{             following conditions are met:                                 }
{                                                                           }
{             * Redistributions of source code must retain the above        }
{               copyright notice, this list of conditions and the following }
{               disclaimer.                                                 }
{             * Redistributions in binary form must reproduce the above     }
{               copyright notice, this list of conditions and the following }
{               disclaimer in the documentation and/or other materials      }
{               provided with the distribution.                             }
{                                                                           }
{               THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER "AS IS"   }
{               AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT   }
{               LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND   }
{               FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO      }
{               EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE   }
{               FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,   }
{               OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,    }
{               PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,   }
{               DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED  }
{               AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT }
{               LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)      }
{               ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF }
{               ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                  }
{                                                                           }

unit Decimals_nasm;

{$IF CompilerVersion >= 22.0}
{$CODEALIGN 16}
{$IFEND}

interface

uses
  SysUtils, Types, Math;

type
  TFormatRec = packed record
    Exponent: Smallint;
    Negative: Boolean;
    Digits: array[0..60] of WideChar;
  end;

  Decimal = packed record
  public
    class var MinValue: Decimal;
    class var MaxValue: Decimal;
    class var One: Decimal;
    class var Zero: Decimal;
    class var MinusOne: Decimal;
    class var OneTenth: Decimal;
    class var Pi: Decimal;
  public
    constructor Create(From: array of Longword); overload;
    constructor Create(From: Currency); overload;
    constructor Create(From: Extended); overload;
    constructor Create(From: Int64); overload;
    constructor Create(From: Longint); overload;
    constructor Create(From: Longword); overload;
    constructor Create(From: UInt64); overload;
    constructor Create(Lo, Mid, Hi: Longword; IsNegative: Boolean; scale: Byte); overload;

    function CompareTo(const D: Decimal): TValueSign;
    function Equals(const D: Decimal): Boolean; overload;
    function GetBytes: TBytes;
    function IsNegative: Boolean; overload;
    function IsZero: Boolean; overload;
    function ToCurrency: Currency;
    function ToDouble: Double;
    function ToExtended: Extended;
    function ToInt64: Int64;
    function ToLongint: Longint;
    function ToLongword: Longword;
    function ToSingle: Single;
    function ToString(const Settings: TFormatSettings): string; overload;
    function ToString(Format: string): string; overload;
    function ToString(Format: string; const Settings: TFormatSettings): string; overload;
    function ToString: string; overload;
    function ToUInt64: UInt64;

    class function Abs(const D: Decimal): Decimal; static;
    class function Add(const Left, Right: Decimal): Decimal; static;
    class function Ceiling(const D: Decimal): Decimal; static;
    class function Compare(const Left, Right: Decimal): TValueSign; static;
    class function Divide(const Left, Right: Decimal): Decimal; static;
    class function Equals(const Left, Right: Decimal): Boolean; overload; static;
    class function Floor(const D: Decimal): Decimal; static;
    class function Frac(const D: Decimal): Decimal; static;
    class function GetBits(const D: Decimal): TLongwordDynArray; static;
    class function Multiply(const Left, Right: Decimal): Decimal; static;
    class function Negate(const D: Decimal): Decimal; static;
    class function Parse(const S: string): Decimal; overload; static;
    class function Parse(const S: string; Settings: TFormatSettings): Decimal; overload; static;
    class function PowerOfTen(Power: Shortint): Decimal; static;
    class function Reciprocal(const D: Decimal): Decimal; static;
    class function Remainder(const Left, Right: Decimal): Decimal; static;
    class function Round(const D: Decimal): Decimal; static;
    class function RoundTo(const D: Decimal; Digits: Integer): Decimal; static;
    class function Sqr(const D: Decimal): Decimal; static;
    class function Subtract(const Left, Right: Decimal): Decimal; static;
    class function Truncate(const D: Decimal): Decimal; static;
    class function TryParse(const S: string; const Settings: TFormatSettings; out Res: Decimal): Boolean; overload; static;
    class function TryParse(const S: string; out Res: Decimal): Boolean; overload; static;

    class operator Add(const Left, Right: Decimal): Decimal;
    class operator Dec(const D: Decimal): Decimal;
    class operator Divide(const Left, Right: Decimal): Decimal;
    class operator Equal(const Left, Right: Decimal): Boolean;
    class operator GreaterThan(const Left, Right: Decimal): Boolean;
    class operator GreaterThanOrEqual(const Left, Right: Decimal): Boolean;
    class operator Implicit(const C: Cardinal): Decimal;
    class operator Implicit(const c: Currency): Decimal;
    class operator Implicit(const D: Decimal): Currency;
    class operator Implicit(const D: Decimal): Extended;
    class operator Implicit(const D: Decimal): Int64;
    class operator Implicit(const D: Decimal): Longint;
    class operator Implicit(const D: Decimal): Longword;
    class operator Implicit(const D: Decimal): string;
    class operator Implicit(const D: Decimal): UInt64;
    class operator Implicit(const D: Double): Decimal;
    class operator Implicit(const E: Extended): Decimal;
    class operator Implicit(const I: Integer): Decimal;
    class operator Implicit(const I64: Int64): Decimal;
    class operator Implicit(const S: Single): Decimal;
    class operator Implicit(const S: string): Decimal;
    class operator Implicit(const UI64: UInt64): Decimal;
    class operator Inc(const D: Decimal): Decimal;
    class operator LessThan(const Left, Right: Decimal): Boolean;
    class operator LessThanOrEqual(const Left, Right: Decimal): Boolean;
    class operator Modulus(const Dividend, Divisor: Decimal): Decimal;
    class operator Multiply(const Left, Right: Decimal): Decimal;
    class operator Negative(const D: Decimal): Decimal;
    class operator NotEqual(const Left, Right: Decimal): Boolean;
    class operator Positive(const D: Decimal): Decimal;
    class operator Round(const D: Decimal): Decimal;
    class operator Subtract(const Left, Right: Decimal): Decimal;
    class operator Trunc(const D: Decimal): Decimal;
{$IFNDEF DEBUG}
  private
{$ENDIF}
{$IF CompilerVersion >= 21.0}
    class constructor CreateRecord;
{$IFEND}
    function ToStringWithSettings(const Settings: TFormatSettings): string;
    function ToStringBlank: string;
    class function TryParseWithSettings(const S: string; const Settings: TFormatSettings;
      out Res: Decimal): Boolean; static;
    class function Div192by32(Quotient: PLongword; const Dividend: Decimal; Divisor: Longword): Longword; static;
    class function Div192by64(Quotient: PLongword; const Dividend, Divisor: Decimal): Longword; static;
    class function Div192by96(Quotient: PLongword; const Dividend, Divisor: Decimal): Longword; static;
    class function Mod96by5(const D: Decimal): Integer; static;
    class procedure InternalAdd(var Result: Decimal; const Left, Right: Decimal); static;
    class procedure InternalAddSub(var Result: Decimal; const Left, Right: Decimal; Sign: Byte); static;
    class procedure InternalCeiling(var Result: Decimal; const Source: Decimal); static;
    class procedure InternalDivide(var Result: Decimal; const Left, Right: Decimal); static;
    class procedure InternalFloor(var Result: Decimal; const Source: Decimal); static;
    class procedure InternalFromDouble(out Result: Decimal; const Source: Double); static;
    class procedure InternalFromExtended(out Result: Decimal; const Source: Extended); static;
    class procedure InternalFromSingle(out Result: Decimal; const Source: Single); static;
    class procedure InternalMultiply(var Result: Decimal; const Left, Right: Decimal); static;
    class procedure InternalRound(var Result: Decimal; const Source: Decimal); static;
    class procedure InternalSubtract(var Result: Decimal; const Left, Right: Decimal); static;
    class procedure InternalToBuffer(var Result: TFormatRec; const Source: Decimal); static;
    class procedure InternalToDouble(var Result: Double; const Source: Decimal); static;
    class procedure InternalToExtended(var Result: Extended; const Source: Decimal); static;
    class procedure InternalToSingle(var Result: Single; const Source: Decimal); static;
    class procedure InternalTruncate(var Result: Decimal; const Source: Decimal); static;
  private
    Lo: Longword;               // Hi:Mid:Lo form 96 bit unsigned mantissa
    Mid: Longword;
    Hi: Longword;
    case Byte of
      0: (Reserved: Word;       // always 0
          Scale: Shortint;      // 0..28
          Sign: Byte);          // $80 = negative, $00 = positive
      1: (Flags: Longword);
  end;

function StrToDecimal(const S: string): Decimal;
function DecimalToStr(const D: Decimal): string;

var
  DoDump: Boolean = False;

implementation

{$POINTERMATH ON}

{$L 'decimals.obj'}

uses
  Windows;

resourcestring
  SDecimalOverflow     = 'Decimal overflow';
  SDecimalUnderflow    = 'Decimal underflow';
  SDecimalZeroDivide   = 'Decimal division by zero';
  SDecimalInvalidOp    = 'Invalid decimal operation';
  SErrorDecimalParsing = '''%S'' is not a valid decimal value';
  SConversionFailed    = 'Decimal value too large for conversion to %S';
  SInvalidArg          = 'Invalid argument: %S';
  SNan                 = 'Invalid argument: NaN (not a number)';
  SEmptyString         = 'empty string';

const
  SingleMShift   = 8;
  SingleMMask    = $007FFFFF;
  SingleEShift   = 23;
  SingleEMask    = $FF;
  SingleBias     = $7F;

  DoubleMShift   = 11;
  DoubleMMask    = $000FFFFF;
  DoubleEShift   = 52;
  DoubleEMask    = $7FF;
  DoubleBias     = $3FF;

  ExtendedEShift = 64;
  ExtendedEMask  = $7FFF;
  ExtendedBias   = $3FFF;

  CMaxValue: Decimal = (Lo: $FFFFFFFF; Mid: $FFFFFFFF; Hi: $FFFFFFFF; Scale:  0; Sign: $00);
  CMinValue: Decimal = (Lo: $FFFFFFFF; Mid: $FFFFFFFF; Hi: $FFFFFFFF; Scale:  0; Sign: $80);
  CMinusOne: Decimal = (Lo: $00000001; Mid: $00000000; Hi: $00000000; Scale:  0; Sign: $80);
  CZero: Decimal     = (Lo: $00000000; Mid: $00000000; Hi: $00000000; Scale:  0; Sign: $00);
  COne: Decimal      = (Lo: $00000001; Mid: $00000000; Hi: $00000000; Scale:  0; Sign: $00);
  COneTenth: Decimal = (Lo: $00000001; Mid: $00000000; Hi: $00000000; Scale:  1; Sign: $00);
  CPi: Decimal       = (Lo: $41B65F29; Mid: $0B143885; Hi: $6582A536; Scale: 28; Sign: $00);

  HLW = High(Longword);
  HUI64 = High(UInt64);
  HLWdiv10 = HLW div 10;
  HUI64div10 = Longword(HUI64 div 10);

  PowersOfTen: array[0..9] of Longword =
  (
             1,
            10,
           100,
          1000,
         10000,
        100000,
       1000000,
      10000000,
     100000000,
    1000000000
  );

  MaxMultiplicands: array[0..9] of Longword =
  (
    HLW,
    HLW div 10,
    HLW div 100,
    HLW div 1000,
    HLW div 10000,
    HLW div 100000,
    HLW div 1000000,
    HLW div 10000000,
    HLW div 100000000,
    HLW div 1000000000
  );

  MaxMultiplicandsMid: array[0..9] of Longword =
  (
    Longword(HUI64),
    Longword(HUI64 div 10),
    Longword(HUI64 div 100),
    Longword(HUI64 div 1000),
    Longword(HUI64 div 10000),
    Longword(HUI64 div 100000),
    Longword(HUI64 div 1000000),
    Longword(HUI64 div 10000000),
    Longword(HUI64 div 100000000),
    Longword(HUI64 div 1000000000)
  );

  DecimalPowersOfTen: array[-28..28] of Decimal =
  (
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $001C0000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $001B0000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $001A0000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00190000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00180000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00170000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00160000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00150000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00140000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00130000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00120000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00110000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00100000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $000F0000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $000E0000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $000D0000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $000C0000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $000B0000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $000A0000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00090000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00080000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00070000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00060000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00050000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00040000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00030000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00020000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00010000),
    (Lo: $00000001; Mid: $00000000; Hi: $00000000; Flags: $00000000),
    (Lo: $0000000A; Mid: $00000000; Hi: $00000000; Flags: $00000000),
    (Lo: $00000064; Mid: $00000000; Hi: $00000000; Flags: $00000000),
    (Lo: $000003E8; Mid: $00000000; Hi: $00000000; Flags: $00000000),
    (Lo: $00002710; Mid: $00000000; Hi: $00000000; Flags: $00000000),
    (Lo: $000186A0; Mid: $00000000; Hi: $00000000; Flags: $00000000),
    (Lo: $000F4240; Mid: $00000000; Hi: $00000000; Flags: $00000000),
    (Lo: $00989680; Mid: $00000000; Hi: $00000000; Flags: $00000000),
    (Lo: $05F5E100; Mid: $00000000; Hi: $00000000; Flags: $00000000),
    (Lo: $3B9ACA00; Mid: $00000000; Hi: $00000000; Flags: $00000000),
    (Lo: $540BE400; Mid: $00000002; Hi: $00000000; Flags: $00000000),
    (Lo: $4876E800; Mid: $00000017; Hi: $00000000; Flags: $00000000),
    (Lo: $D4A51000; Mid: $000000E8; Hi: $00000000; Flags: $00000000),
    (Lo: $4E72A000; Mid: $00000918; Hi: $00000000; Flags: $00000000),
    (Lo: $107A4000; Mid: $00005AF3; Hi: $00000000; Flags: $00000000),
    (Lo: $A4C68000; Mid: $00038D7E; Hi: $00000000; Flags: $00000000),
    (Lo: $6FC10000; Mid: $002386F2; Hi: $00000000; Flags: $00000000),
    (Lo: $5D8A0000; Mid: $01634578; Hi: $00000000; Flags: $00000000),
    (Lo: $A7640000; Mid: $0DE0B6B3; Hi: $00000000; Flags: $00000000),
    (Lo: $89E80000; Mid: $8AC72304; Hi: $00000000; Flags: $00000000),
    (Lo: $63100000; Mid: $6BC75E2D; Hi: $00000005; Flags: $00000000),
    (Lo: $DEA00000; Mid: $35C9ADC5; Hi: $00000036; Flags: $00000000),
    (Lo: $B2400000; Mid: $19E0C9BA; Hi: $0000021E; Flags: $00000000),
    (Lo: $F6800000; Mid: $02C7E14A; Hi: $0000152D; Flags: $00000000),
    (Lo: $A1000000; Mid: $1BCECCED; Hi: $0000D3C2; Flags: $00000000),
    (Lo: $4A000000; Mid: $16140148; Hi: $00084595; Flags: $00000000),
    (Lo: $E4000000; Mid: $DCC80CD2; Hi: $0052B7D2; Flags: $00000000),
    (Lo: $E8000000; Mid: $9FD0803C; Hi: $033B2E3C; Flags: $00000000),
    (Lo: $10000000; Mid: $3E250261; Hi: $204FCE5E; Flags: $00000000)
  );

type
  TAccumulator = packed record
    case Byte of
      0: (L0, L1, L2, L3, L4, L5, L6: Longword;
          Flags: Longword;
          Underflow: Longword);
      1: (Bits: array [0..6] of Longword;
          Reserved: Word;
          Scale: Shortint;
          Sign: ByteBool);
  end;

  TUnsigned64 = packed record
    case Byte of
      0: (UI64: UInt64);
      1: (Lo32: Longword;
          Hi32: Longword);
  end;

type
  TDecimalErrorType = (detOverflow, detUnderflow, detZeroDivide, detInvalidOp,
    detParse, detConversion, detInvalidArg, detNaN);

procedure DumpAccu(const A: TAccumulator);
begin
  if DoDump then
    Writeln(Format('Accu: %.2d %.8x %.8x %.8x %.8x %.8x %.8x', [A.Scale, A.L5, A.L4, A.L3, A.L2, A.L1, A.L0]));
end;

procedure Error(N: TDecimalErrorType; const S: string = '');
begin
  case N of
    detOverflow:
      raise EOverflow.Create(SDecimalOverflow);
    detUnderflow:
      raise EUnderflow.Create(SDecimalUnderflow);
    detZeroDivide:
      raise EZeroDivide.Create(SDecimalZeroDivide);
    detParse:
      raise EConvertError.CreateFmt(SErrorDecimalParsing, [S]);
    detConversion:
      raise EConvertError.CreateFmt(SConversionFailed, [S]);
    detInvalidArg:
      raise EInvalidArgument.CreateFmt(SInvalidArg, [S]);
    detNaN:
      raise EInvalidArgument.Create(SNaN);
  else
    raise EInvalidOp.Create(SDecimalInvalidOp);
  end;
end;

function StrToDecimal(const S: string): Decimal;
begin
  if not Decimal.TryParse(S, Result) then
    Error(detParse, S);
end;

function DecimalToStr(const D: Decimal): string;
begin
  Result := D.ToString;
end;

// Scales decimal down, i.e. divides Hi:Mid:Lo by 10 and decrements Scale
// without any integrity checks.  Returns remainder of division (0..9).
function DecScaleDownRaw(var D: Decimal): Byte; external;

// Scales decimal down, i.e. divides "mantissa" by powers of 10 and decrements
// Scale, without any integrity checks. Rounds Result.
procedure DecScaleDownMaxRaw(var D: Decimal; Max: Integer); external;

procedure DecScaleDownTo(var D: Decimal; TargetScale: Byte); external;

// Scales decimal up, i.e. multiplies "mantissa" by 10 and increments Scale,
// without any integrity checks.  Returns any EDX overflow.
function DecScaleUpRaw(var D: Decimal): Longword; external;

function DecScaleUpMax(var D: Decimal; Max: Integer): Boolean; external;

// Scales decimal up, i.e. multiplies "mantissa" by 10 and increments Scale,
// then adds Num to "mantissa", without any integrity checks.
// Returns any overflow.
function DecScaleUpAndAdd(var D: Decimal; Num: Byte): Longword; external;

// Scales accumulator down, i.e. divides it by 10 and increments Scale.
// No integrity checks.
procedure AccuScaleDownRaw(var A: TAccumulator); external;

procedure AccuScaleDownRaw128(var A: TAccumulator); external;

procedure AccuScaleDownMax(var A: TAccumulator; Max: Integer); external;

procedure AccuScaleDownMax128(var A: TAccumulator; Max: Integer); external;

// Scales accumulator up, i.e. multiplies it by 10 and increments Scale.
// Returns any overflow.
procedure AccuScaleUp(var A: TAccumulator); external;

// Scales accumulator up, i.e. multiplies it by powers of 10 and increments
// Scale accordingly. Performs integrity checks.
// Returns any overflow.
function AccuScaleUpMax(var A: TAccumulator; Max: Integer): Longword; external;

function AccuScaleUpMax224(var A: TAccumulator; Max: Integer): Longword; external;

{ Decimal }

class function Decimal.Abs(const D: Decimal): Decimal;
begin
  if @Result <> @D then
    Result := D;
  Result.Sign := 0;
end;

class function Decimal.Add(const Left, Right: Decimal): Decimal;
begin
  InternalAdd(Result, Left, Right);
end;

class operator Decimal.Add(const Left, Right: Decimal): Decimal;
begin
  InternalAdd(Result, Left, Right);
end;

class function Decimal.Ceiling(const D: Decimal): Decimal;
begin
  InternalCeiling(Result, D);
end;

class function Decimal.Compare(const Left, Right: Decimal): TValueSign; external;

function Decimal.CompareTo(const D: Decimal): TValueSign;
begin
  Result := Decimal.Compare(Self, D);
end;

constructor Decimal.Create(From: Int64);
var
  Sign: Boolean;
begin
  Sign := From < 0;
  if Sign then
    From := -From;
  Create(From and $FFFFFFFF, From shr 32, 0, Sign, 0);
end;

constructor Decimal.Create(From: UInt64);
begin
  Create(From and $FFFFFFFF, From shr 32, 0, False, 0);
end;

constructor Decimal.Create(From: Integer);
var
  Sign: Boolean;
begin
  Sign := From < 0;
  if Sign then
    From := -From;
  Create(From, 0, 0, Sign, 0);
end;

constructor Decimal.Create(From: Longword);
begin
  Create(From, 0, 0, False, 0);
end;

constructor Decimal.Create(From: Extended);
begin
  InternalFromExtended(Self, From);
end;

constructor Decimal.Create(From: array of Longword);
begin
  if Length(From) = 4 then
  begin
    Lo := From[0];
    Mid := From[1];
    Hi := From[2];
    Flags := From[3];
  end;
end;

constructor Decimal.Create(Lo, Mid, Hi: Longword; IsNegative: Boolean;
  scale: Byte);
begin
  Self.Lo := Lo;
  Self.Mid := Mid;
  Self.Hi := Hi;
  Flags := 0;
  if IsNegative then
    Sign := $80;
  Self.Scale := scale;
end;

constructor Decimal.Create(From: Currency);
var
  Sign: Boolean;
begin
  Sign := From < 0;
  if Sign then
    From := -From;
  Create(PLongword(@From)[0], PLongword(@From)[1], 0, Sign, 4);
end;

class operator Decimal.Dec(const D: Decimal): Decimal;
begin
  Result := D + Decimal.MinusOne;
end;

class function Decimal.Divide(const Left, Right: Decimal): Decimal;
begin
  InternalDivide(Result, Left, Right);
end;

class operator Decimal.Divide(const Left, Right: Decimal): Decimal;
begin
  InternalDivide(Result, Left, Right);
end;

class operator Decimal.Equal(const Left, Right: Decimal): Boolean;
begin
  Result := Compare(Left, Right) = 0;
end;

function Decimal.Equals(const D: Decimal): Boolean;
begin
  Result := Decimal.Compare(Self, D) = 0;
end;

class function Decimal.Equals(const Left, Right: Decimal): Boolean;
begin
  Result := Decimal.Compare(Left, Right) = 0;
end;

class function Decimal.Floor(const D: Decimal): Decimal;
begin
  InternalFloor(Result, D);
end;

class function Decimal.Frac(const D: Decimal): Decimal;
begin
  Result := D - Decimal.Truncate(D);
end;

class procedure Decimal.InternalFromDouble(out Result: Decimal; const Source: Double); external;

class procedure Decimal.InternalFromExtended(out Result: Decimal; const Source: Extended); external;

class procedure Decimal.InternalFromSingle(out Result: Decimal; const Source: Single); external;

class function Decimal.GetBits(const D: Decimal): TLongwordDynArray;
begin
  SetLength(Result, SizeOf(Decimal) div SizeOf(Longword));
  Move(D, Result[0], SizeOf(Decimal));
end;

function Decimal.GetBytes: TBytes;
begin
  SetLength(Result, SizeOf(Decimal));
  Move(Self, Result[0], SizeOf(Decimal));
end;

class operator Decimal.GreaterThan(const Left, Right: Decimal): Boolean;
begin
  Result := Compare(Left, Right) > 0;
end;

class operator Decimal.GreaterThanOrEqual(const Left, Right: Decimal): Boolean;
begin
  Result := Compare(Left, Right) >= 0;
end;

class operator Decimal.Implicit(const C: Cardinal): Decimal;
begin
  Result := Decimal.Create(C);
end;

class operator Decimal.Implicit(const UI64: UInt64): Decimal;
begin
  Result := Decimal.Create(UI64);
end;

class operator Decimal.Implicit(const S: Single): Decimal;
begin
  InternalFromSingle(Result, S);
end;

class operator Decimal.Implicit(const D: Decimal): Int64;
begin
  Result := D.ToInt64;
end;

class operator Decimal.Implicit(const D: Decimal): Extended;
begin
  Result := D.ToExtended;
end;

class operator Decimal.Implicit(const D: Decimal): Currency;
begin
  Result := D.ToCurrency;
end;

class operator Decimal.Implicit(const D: Decimal): UInt64;
begin
  Result := D.ToUInt64;
end;

class operator Decimal.Implicit(const D: Decimal): Longword;
begin
  Result := D.ToLongword;
end;

class operator Decimal.Implicit(const D: Decimal): Longint;
begin
  Result := D.ToLongint;
end;

class operator Decimal.Implicit(const D: Decimal): string;
begin
  Result := D.ToString;
end;

class operator Decimal.Implicit(const I: Integer): Decimal;
begin
  Result := Decimal.Create(I);
end;

class operator Decimal.Implicit(const I64: Int64): Decimal;
begin
  Result := Decimal.Create(I64);
end;

class operator Decimal.Implicit(const E: Extended): Decimal;
begin
  InternalFromExtended(Result, E);
end;

class operator Decimal.Implicit(const D: Double): Decimal;
begin
  InternalFromDouble(Result, D);
end;

class operator Decimal.Implicit(const S: string): Decimal;
begin
  Result := Decimal.Parse(S);
end;

class operator Decimal.Implicit(const c: Currency): Decimal;
begin
  Result := Decimal.Create(c);
end;

class operator Decimal.Inc(const D: Decimal): Decimal;
begin
  Result := D + Decimal.One;
end;

class procedure Decimal.InternalAdd(var Result: Decimal; const Left, Right: Decimal);
begin
  InternalAddSub(Result, Left, Right, 0);
end;

class procedure Decimal.InternalAddSub(var Result: Decimal; const Left, Right: Decimal; Sign: Byte);
  external;

class procedure Decimal.InternalCeiling(var Result: Decimal; const Source: Decimal); external;

class function Decimal.Div192by32(Quotient: PLongword; const Dividend: Decimal;
  Divisor: Longword): Longword; external;

class function Decimal.Div192by64(Quotient: PLongword; const Dividend, Divisor: Decimal): Longword;
  external;

class function Decimal.Div192by96(Quotient: PLongword; const Dividend, Divisor: Decimal): Longword;
  external;

class function Decimal.Mod96by5(const D: Decimal): Integer; external;

class procedure Decimal.InternalDivide(var Result: Decimal; const Left, Right: Decimal); external;

class procedure Decimal.InternalFloor(var Result: Decimal; const Source: Decimal); external;

// Basecase multiplication
//
// Karatsuba, Divide and Conquer, etc. are probably only useful for larger sizes.

class procedure Decimal.InternalMultiply(var Result: Decimal;
  const Left, Right: Decimal); external;

class procedure Decimal.InternalRound(var Result: Decimal; const Source: Decimal); external;

class procedure Decimal.InternalSubtract(var Result: Decimal;
  const Left, Right: Decimal);
begin
  InternalAddSub(Result, Left, Right, $80);
end;

class procedure Decimal.InternalToBuffer(var Result: TFormatRec; const Source: Decimal);
begin

end;

class procedure Decimal.InternalToDouble(var Result: Double; const Source: Decimal); external;

class procedure Decimal.InternalToExtended(var Result: Extended; const Source: Decimal); external;

class procedure Decimal.InternalToSingle(var Result: Single; const Source: Decimal); external;

class procedure Decimal.InternalTruncate(var Result: Decimal; const Source: Decimal); external;

function Decimal.IsNegative: Boolean;
begin
  Result := Sign <> 0;
end;

function Decimal.IsZero: Boolean;
begin
  Result := (Lo or Mid or Hi) = 0;
end;

class operator Decimal.LessThan(const Left, Right: Decimal): Boolean;
begin
  Result := Compare(Left, Right) < 0;
end;

class operator Decimal.LessThanOrEqual(const Left, Right: Decimal): Boolean;
begin
  Result := Compare(Left, Right) <= 0;
end;

class function Decimal.Multiply(const Left, Right: Decimal): Decimal;
begin
  InternalMultiply(Result, Left, Right);
end;

class operator Decimal.Multiply(const Left, Right: Decimal): Decimal;
begin
  InternalMultiply(Result, Left, Right);
end;

class operator Decimal.Modulus(const Dividend, Divisor: Decimal): Decimal;
var
  d1, d2, d4: Decimal;

begin
  d1 := Dividend;
  d2 := Divisor;

  d2.Sign := d1.Sign;
  if Decimal.Abs(d1) < Decimal.Abs(d2) then
    Exit(d1);
  d1 := d1 - d2;
  if d1 = Decimal.Zero then
    d1.Sign := d2.Sign;
  d4 := d1 - Truncate(d1 / d2) * d2;
  if d1.Sign = d4.Sign then
    Exit(d4);
  if d4 = Decimal.Zero then
  begin
    d4.Sign := d1.Sign;
    Exit(d4);
  end;
  Exit(d4 + d2);

//  Result := Dividend - Trunc(Dividend / Divisor) * Divisor;
end;

class function Decimal.Negate(const D: Decimal): Decimal;
begin
  Result := D;
  Result.Sign := D.Sign xor $80;
end;

class operator Decimal.Negative(const D: Decimal): Decimal;
begin
  Result := D;
  Result.Sign := D.Sign xor $80;
end;

class operator Decimal.NotEqual(const Left, Right: Decimal): Boolean;
begin
  Result := Compare(Left, Right) <> 0;
end;

class function Decimal.Parse(const S: string; Settings: TFormatSettings): Decimal;
begin
  if not TryParse(S, Settings, Result) then
    Error(detParse, S);
end;

class function Decimal.Parse(const S: string): Decimal;
begin
  if not TryParse(S, Result) then
    Error(detParse, S);
end;

class operator Decimal.Positive(const D: Decimal): Decimal;
begin
  Result := D;
end;

class function Decimal.PowerOfTen(Power: Shortint): Decimal;
begin
  if (Power < -28) or (Power > 28) then
    Error(detInvalidArg, 'power');
  Result := DecimalPowersOfTen[Power];
end;

class function Decimal.Reciprocal(const D: Decimal): Decimal;
begin
  Result := Decimal.One / D;
end;

class function Decimal.Remainder(const Left, Right: Decimal): Decimal;
begin
  Result := Left mod Right;
end;

class function Decimal.Round(const D: Decimal): Decimal;
begin
  InternalRound(Result, D);
end;

class operator Decimal.Round(const D: Decimal): Decimal;
begin
  InternalRound(Result, D);
end;

class function Decimal.RoundTo(const D: Decimal; Digits: Integer): Decimal;
begin
  if (Digits < -28) or (Digits > 28) then
    Error(detInvalidArg, 'digits');
  Result := D;
  if Digits >= 0 then
  begin
    Result := D;
    while Result.Scale > Digits do
      DecScaleDownMaxRaw(Result, Result.Scale - Digits);
    while (Result.Scale < Digits) and DecScaleUpMax(Result, Digits - Result.Scale) do
      ;
  end
  else
  begin
    InternalMultiply(Result, D, DecimalPowersOfTen[Digits]);
    InternalRound(Result, Result);
    InternalMultiply(Result, Result, DecimalPowersOfTen[-Digits]);
  end;
end;

class function Decimal.Sqr(const D: Decimal): Decimal;
begin
  InternalMultiply(Result, D, D);
end;

class operator Decimal.Subtract(const Left, Right: Decimal): Decimal;
begin
  InternalSubtract(Result, Left, Right);
end;

class function Decimal.Subtract(const Left, Right: Decimal): Decimal;
begin
  InternalSubtract(Result, Left, Right);
end;

function Decimal.ToCurrency: Currency;
var
  D: Decimal;
begin
  try
    D := Decimal.RoundTo(Self, 4);
  except
    Error(detConversion, 'Currency');
  end;
  if (D.Hi or (D.Mid and $80000000)) <> 0 then
    Error(detConversion, 'Currency');
  PLongword(@Result)[0] := D.Lo;
  PLongword(@Result)[1] := D.Mid;
  if Sign <> 0 then
    Result := -Result;
end;

function Decimal.ToDouble: Double;
begin
  InternalToDouble(Result, Self);
end;

function Decimal.ToExtended: Extended;
begin
  InternalToExtended(Result, Self);
end;

function Decimal.ToInt64: Int64;
var
  D: Decimal;
begin
  D := Decimal.Round(Self);
  if (D.Hi or (D.Mid and $80000000)) <> 0 then
    Error(detConversion, 'Int64');
  PLongword(@Result)[0] := D.Lo;
  PLongword(@Result)[1] := D.Mid;
  if Sign <> 0 then
    Result := -Result;
end;

function Decimal.ToLongint: Longint;
var
  D: Decimal;
begin
  D := Decimal.Round(Self);
  if (D.Hi or D.Mid or (D.Lo and $80000000)) <> 0 then
    Error(detConversion, 'Longint');
  Longword(Result) := D.Lo;
  if Sign <> 0 then
    Result := -Result;
end;

function Decimal.ToLongword: Longword;
var
  D: Decimal;
begin
  D := Decimal.Round(Self);
  if (D.Hi or D.Mid) <> 0 then
    Error(detConversion, 'Longword');
  Longword(Result) := D.Lo;
end;

procedure StoreChar(var P: PChar; C: Char); inline;
begin
  P^ := C;
  Dec(P);
end;

(*
Format string output:

Format string                       Output                                          Type
----------------------------------- ----------------------------------------------- ----------------------
<none>                              123456789,123456789                             unformatted
<empty>                             123456789,123456789                             global (G)
C:                                  123.456.789,12 ?                                currency (fff.fff.fff,ff $)
C20:                                123.456.789,12345678900000000000 ?              currency 20 decimals
E:                                  1,234568E+008                                   exponential (1,ffffffE+00n)
E40:                                1,2345678912345678900000000000000000000000E+008 exponential 40 characters
F:                                  123456789,12                                    fixed point (fffff,ff)
F10:                                123456789,1234567890                            fixed point 10 decimals
G:                                  123456789,123456789                             general
G10:                                123456789,1                                     general 10 characters
N:                                  123.456.789,12                                  number (2 decimals)
N10:                                123.456.789,1234567890                          number 10 decimals
P:                                  12.345.678.912,35%                              percent (2 decimals)
P10:                                12.345.678.912,3456789000%                      percent 10 decimals
Cuckoo 0000000000.000000000000000:  Cuckoo 0123456789,123456789000000
Cuckoo 0,00000000.000000000000000:  Cuckoo 123.456.789,123456789000000
Cuckoo 0000000000,000000000000000:  Cuckoo 0.000.000.000.000.000.123.456.789
Cuckoo 0000000000,000000000.000000: Cuckoo 0.000.000.000.123.456.789,123457

See:
ms-help://MS.NETFramework.v20.en/dv_fxfund/html/580e57eb-ac47-4ffd-bccd-3a1637c2f467.htm (standard)
ms-help://MS.NETFramework.v20.en/dv_fxfund/html/6f74fd32-6c6b-48ed-8241-3c2b86dea5f4.htm (custom)
ms-help://MS.NETFramework.v20.en/dv_fxfund/html/33f27256-722f-4296-a969-3a07dd4f2a02.htm (examples of custom)
*)

function Decimal.ToString(Format: string): string;
begin
  // TODO: implement
end;

procedure __UStrFromPWCharLen(var Dest: UnicodeString; Source: PWideChar; CharLength: Integer);
begin
  SetLength(Dest, CharLength);
  StrLCopy(PWideChar(Dest), Source, CharLength);
end;

function Decimal.ToString(const Settings: TFormatSettings): string;
begin
  Result := ToStringWithSettings(Settings);
end;

function Decimal.ToStringWithSettings(const Settings: TFormatSettings): string; external;

function Decimal.ToString(Format: string; const Settings: TFormatSettings): string;
begin
  // TODO: implement
end;

function Decimal.ToSingle: Single;
begin
  InternalToSingle(Result, Self);
end;

function Decimal.ToString: string;
begin
  Result := ToStringBlank;
end;

function Decimal.ToStringBlank: string; external;

function Decimal.ToUInt64: UInt64;
var
  D: Decimal;
begin
  D := Decimal.Round(Self);
  if D.Hi <> 0 then
    Error(detConversion, 'UInt64');
  PLongword(@Result)[0] := D.Lo;
  PLongword(@Result)[1] := D.Mid;
end;

class operator Decimal.Trunc(const D: Decimal): Decimal;
begin
  InternalTruncate(Result, D);
end;

class function Decimal.Truncate(const D: Decimal): Decimal;
begin
  InternalTruncate(Result, D);
end;

class function Decimal.TryParse(const S: string;
  const Settings: TFormatSettings; out Res: Decimal): Boolean;
begin
  Result := TryParseWithSettings(S, Settings, Res);
end;

// Tries to parse S as Decimal. Returns True if successful.
class function Decimal.TryParseWithSettings(const S: string;
  const Settings: TFormatSettings; out Res: Decimal): Boolean; external;

class function Decimal.TryParse(const S: string; out Res: Decimal): Boolean;
var
  Settings: TFormatSettings;
begin
  Settings := TFormatSettings.Create(LOCALE_SYSTEM_DEFAULT);
  Result := TryParse(S, Settings, Res);
end;

procedure InitClassVars;
begin
  Decimal.MaxValue := CMaxValue;
  Decimal.MinValue := CMinValue;
  Decimal.MinusOne := CMinusOne;
  Decimal.Zero := CZero;
  Decimal.One := COne;
  Decimal.OneTenth := COneTenth;
  Decimal.Pi := CPi;
end;

{$IF CompilerVersion >= 21.0}
class constructor Decimal.CreateRecord;
begin
  InitClassVars;
end;
{$ELSE}
initialization
  InitClassVars;
{$IFEND}

end.
