program TestStringPos;

{$APPTYPE CONSOLE}

{$R *.res}

uses
  System.SysUtils,
  System.Diagnostics,
  System.StrUtils,
  Velthuis.AutoConsole,
  System.Generics.Collections,
  System.AnsiStrings,
  TestPosTools in 'TestPosTools.pas',
  PosProcs in 'PosProcs.pas';

const
  HLen = 500*1000*1000;
  CLoops = 2000;

{$OPTIMIZATION ON}

type
  TUTestProc = function(const Needle, Haystack: UnicodeString; Offset: Integer): Integer;
  TATestProc = function(const Needle, Haystack: AnsiString; Offset: Integer): Integer;
  TBTestProc = function(const Needle, Haystack: TBytes; Offset: Integer): Integer;

procedure Caption(const S: string);
begin
  Writeln(S);
  Writeln(StringofChar('-', Length(S)));
end;

procedure TimeUProc(const Name: string; Proc: TUTestProc; const HayStacks, Needles: array of UnicodeString; Loops: Integer);
var
  I, J, K: Integer;
  SW: TStopwatch;
begin
  SW := TStopwatch.StartNew;
  for K := 1 to Loops do
    for I := Low(Haystacks) to High(Haystacks) do
      for J := Low(Needles) to High(Needles) do
        Proc(Needles[J], Haystacks[I], 1);
  Writeln(Format('%-20s %5d ms', [Name + ':', SW.ElapsedMilliseconds]));
end;

procedure TimeAProc(const Name: string; Proc: TATestProc; const HayStacks, Needles: array of AnsiString; Loops: Integer);
var
  I, J, K: Integer;
  SW: TStopwatch;
begin
  SW := TStopwatch.StartNew;
  for K := 1 to Loops do
    for I := Low(Haystacks) to High(Haystacks) do
      for J := Low(Needles) to High(Needles) do
        Proc(Needles[J], Haystacks[I], 1);
  Writeln(Format('%-20s %5d ms', [Name + ':', SW.ElapsedMilliseconds]));
end;

procedure TimeBProc(const Name: string; Proc: TBTestProc; const HayStacks, Needles: array of TBytes; Loops: Integer);
var
  I, J, K: Integer;
  SW: TStopwatch;
begin
  SW := TStopwatch.StartNew;
  for K := 1 to Loops do
    for I := Low(Haystacks) to High(Haystacks) do
      for J := Low(Needles) to High(Needles) do
        Proc(Needles[J], Haystacks[I], 1);
  Writeln(Format('%-20s %5d ms', [Name + ':', SW.ElapsedMilliseconds]));
end;

function CallPosA(const ANeedle, AHaystack: AnsiString; Offset: Integer): Integer;
begin
  Result := System.Pos(ANeedle, AHaystack);
end;

procedure TestUAndAProcs;
var
  HaystacksU, NeedlesU: TArray<UnicodeString>;
  HaystacksA, NeedlesA: TArray<AnsiString>;
  HaystacksB, NeedlesB: TArray<TBytes>;
  I: Integer;
begin
  // Set up UnicodeString arrays.
  HaystacksU := TArray<UnicodeString>.Create(RndUString(50), RndUString(200),
    RndUString(3000), RndUString(4000), RndUString(300000));
  NeedlesU := TArray<UnicodeString>.Create(RndUString(1), RndUString(3), RndUString(8), RndUString(20));

  // Copy to AnsiStrings arrays.
  SetLength(HaystacksA, Length(HaystacksU));
  for I := Low(HaystacksU) to High(HaystacksU) do
    HaystacksA[I] := AnsiString(HaystacksU[I]);

  SetLength(NeedlesA, Length(NeedlesU));
  for I := Low(NeedlesU) to High(NeedlesU) do
    NeedlesA[I] := AnsiString(NeedlesU[I]);

  SetLength(NeedlesB, Length(NeedlesA));
  for I := Low(NeedlesA) to High(NeedlesA) do
    NeedlesB[I] := BytesOf(NeedlesA[I]);

  SetLength(HaystacksB, Length(HaystacksA));
  for I := Low(HaystacksA) to High(HaystacksA) do
    HaystacksB[I] := BytesOf(HaystacksA[I]);

  // Timing.
  Writeln('Testing with Haystack lengths of 50, 200, 3000, 4000 and 300000');
  Writeln('and Needle lengths of 1, 3, 8 and 20');
  Writeln(Length(HaystacksU), ' * ', Length(NeedlesU), ' * ', CLoops, ' = ', Length(HaystacksU) * Length(NeedlesU) * CLoops, ' loops');
  Writeln;

  Caption('UnicodeString');
  TimeUProc('System.Pos', System.Pos, HaystacksU, NeedlesU, CLoops);
  TimeUProc('StrUtils.PosEx', System.StrUtils.PosEx, HaystacksU, NeedlesU, CLoops);
  TimeUProc('RVPosExU', RVPosExU, HaystacksU, NeedlesU, CLoops);
  Writeln;
  Caption('AnsiString');
  TimeAProc('System.Pos', CallPosA, HaystacksA, NeedlesA, CLoops);
  TimeAProc('AnsiStrings.PosEx', System.AnsiStrings.PosEx, HaystacksA, NeedlesA, CLoops);
  TimeAProc('OrgPosA', OrgPosA, HaystacksA, NeedlesA, CLoops);
  TimeAProc('PosUModForA', PosUModForA, HaystacksA, NeedlesA, CLoops);
  TimeAProc('RVPosExA', RVPosExA, HayStacksA, NeedlesA, CLoops);
  Writeln;
  Caption('TBytes');
  TimeBProc('RVPosEXB', RVPosExB, HaystacksB, NeedlesB, CLoops);
  Writeln;
end;

type
  TTestProc = System.SysUtils.TProc;

procedure TimeProc(const Name: string; Proc: TTestProc);
var
  SW: TStopwatch;
begin
  SW := TStopwatch.StartNew;
  Proc;
  Writeln(Format('%-20s %5d ms', [Name + ':', SW.ElapsedMilliseconds]));
end;

procedure TestHugeString;
var
  UNeedle, UHaystack: UnicodeString;
  ANeedle, AHaystack: AnsiString;
  BNeedle, BHaystack: TBytes;
begin
  AHaystack := RndAString(HLen);
  ANeedle := Copy(AHaystack, Length(AHaystack) - 9, Maxint);
  UHaystack := UnicodeString(AHaystack);
  UNeedle := UnicodeString(ANeedle);

  Writeln('Haystack: random string of ', HLen, ' ASCII characters or bytes');
  Writeln('Needle: last 10 characters of Haystack = ''', UNeedle, '''');
  Writeln;

  Caption('UnicodeString');
  TimeProc('System.Pos', procedure begin System.Pos(UNeedle, UHaystack) end);
  TimeProc('Strutils.PosEx', procedure begin System.StrUtils.PosEx(UNeedle, UHaystack) end);
  TimeProc('RVPosExU', procedure begin RVPosExU(UNeedle, UHaystack) end);
  Writeln;

  Caption('AnsiString');
  TimeProc('System.Pos', procedure begin System.Pos(ANeedle, AHaystack) end);
  TimeProc('AnsiStrings.PosEx', procedure begin System.AnsiStrings.PosEx(ANeedle, AHaystack) end);
  TimeProc('OrgPosA', procedure begin OrgPosA(ANeedle, AHaystack, 1) end);
  TimeProc('PosUModForA', procedure begin PosUModForA(ANeedle, AHaystack, 1) end);
  TimeProc('RVPosExA', procedure begin RVPosExA(ANeedle, AHaystack, 1) end);
  TimeProc('RvPosExA(,,Offset)', procedure begin RVPosExA(ANeedle, AHaystack, Length(AHaystack) div 2) end);

  // Do not risk an out of memory: delete the looooong strings.
  UNeedle := '';
  UHaystack := '';
  BNeedle := BytesOf(ANeedle);
  ANeedle := '';
  BHaystack := BytesOf(AHaystack);
  AHaystack := '';

  Writeln;
  Caption('TBytes');
  TimeProc('RVPosExB(TBytes)', procedure begin RVPosExB(BNeedle, BHaystack) end);
  BHaystack := nil;
  BNeedle := nil;
  Writeln;
end;

begin
  Writeln;
{$IF defined(WIN32)}
  Writeln('WIN32');
  Writeln;
{$ELSEIF defined(WIN64)}
  Writeln('WIN64');
  Writeln;
{$ELSEIF defined(MACOS32)}
  Writeln('MACOS32');
  Writeln;
{$ENDIF}
  Writeln('Different versions of Pos(Needle, Haystack: <sometype>; Offset: Integer): Integer');
  Writeln('where <sometype> is UnicodeString, AnsiString or TBytes');
  Writeln;
  try
    Randomize;
    TestUAndAProcs;
    Writeln;
    TestHugeString;
  except
    on E: Exception do
      Writeln(E.ClassName, ': ', E.Message);
  end;
end.
