/*
  File:       RadioDial.cpp
  Function:   TRadioDial component, a button like the dial on a radio.
  Language:   BCB3 (C++)
              Translation and enhancement of the original Object Pascal
              version (by same author)
  Author:     Rudy Velthuis
  Copyright:  (c) 1997,2002 drs. Rudolph Velthuis
  Disclaimer: This code is freeware. All rights are reserved.
              This code is provided as is, expressly without a warranty of
              any kind. You use it at your own risk.
              If you use this code, please credit me.
*/

#include <vcl.h>
#pragma hdrstop

#include "RadioDial.h"
#include <Consts.hpp>   // for VCL constants
#include <cmath>        // for sin, cos, atan, floor

#pragma package(smart_init)

// ValidCtrCheck is used to assure that the components created do not have
// any pure virtual functions.

static inline void ValidCtrCheck(TCustomRadioDial *)
{
   new TCustomRadioDial(NULL);
}

const MinBorder = 1;
const TickBorder = TCustomRadioDial::tlLongLen;

// Get current shift state as a TShiftState.
inline static TShiftState GetShiftState(void)
{
   TShiftState Result;

   if (GetAsyncKeyState(VK_SHIFT) < 0) Result << ssShift;
   if (GetAsyncKeyState(VK_CONTROL) < 0) Result << ssCtrl;
   if (GetAsyncKeyState(VK_MENU) < 0) Result << ssAlt;

   return Result;
}


// Virtual constructor.
__fastcall TCustomRadioDial::TCustomRadioDial(TComponent* Owner)
   : inherited(Owner),
     dRadianToAngle(1800.0 / M_PI),
     dAngleToRadian(M_PI / 1800.0)
{
   ControlStyle << csClickEvents << csCaptureMouse;
   FBorderStyle = Radiodial::bsNone;
   FButtonEdge = 5;
   FDefaultPos = 0;
   FFrequency = 10;
   FLargeChange = 2;
   FMax = 100;
   FMaxAngle = 3300;
   FMin = 0;
   FMinAngle = 300;
   FPointerColor = clBtnText;
   FPointerSize = 33;
   FRadius = rcMinRadius;
   FSmallChange = 1;
   TabStop = True;
   FTickStyle = tsAuto;
   FBitmapInvalid = true;
   FPointerRect.Left = -1;      // Only on start up
   Width = 51;
   Height = 51;
   FRepeatDelay = 400;
   FRepeatRate = 100;
   SetTicks(FTickStyle);
   Position = 0;
}


__fastcall TCustomRadioDial::~TCustomRadioDial()
{
   delete FRepeatTimer;
   ClearTicks();
   delete FBitmap;
}

// Convert position Pos to an angle.
TRadioAngle __fastcall TCustomRadioDial::PosToAngle(int Pos)
{
   return TRadioAngle(FMinAngle +
             ((FMaxAngle - FMinAngle) * (Pos - FMin) / (FMax - FMin)));
}

// Convert angle AnAngle to a position.
int __fastcall TCustomRadioDial::AngleToPos(TRadioAngle AnAngle)
{
   return FMin + ((FMax - FMin) * (AnAngle - FMinAngle) / (FMaxAngle - FMinAngle));
}

// Convert polar coordinates defined by AnAngle, ACenter and ARadius to a TPoint.
TPoint __fastcall TCustomRadioDial::AngleToPoint(TRadioAngle AnAngle,
   const TPoint &ACenter, int ARadius)
{
   double radAngle = AngleToRad(AnAngle);
   return Point(std::ceil(ACenter.x - ARadius * std::sin(radAngle)),
                std::ceil(ACenter.y + ARadius * std::cos(radAngle)));
}

// Convert a APoint to an angle (relative to ACenter) in radians, where
// bottom is 0, left is Pi/2, top is Pi and so on.
static double inline PointToRad(const TPoint APoint, const TPoint ACenter)
{
   int n = APoint.x - ACenter.x;
   double result;

   if (n == 0)
      result = 0.5 * M_PI;
   else
      result = std::atan(double(ACenter.y - APoint.y) / double(n));
   if (n < 0)
      result += M_PI;
   return 1.5 * M_PI - result;
}

// Get current angle (from position).
TRadioAngle __fastcall TCustomRadioDial::GetAngle(void)
{
   return PosToAngle(FPosition);
}

// Set current angle. Sets Position.
void __fastcall TCustomRadioDial::SetAngle(TRadioAngle Value)
{
   SetAngleParams(Value, FMinAngle, FMaxAngle);
}

// Set border style. Redraw if necessary.
void __fastcall TCustomRadioDial::SetBorderStyle(TRadioBorderStyle Value)
{
   if (Value != FBorderStyle)
   {
      FBorderStyle = Value;
      if (HandleAllocated())
      {
         RecreateWnd();
         DrawBorder();
      }
   }
}

// Set positional (Cartesian) parameters, value checked and invalidate if
// necessary.
void __fastcall TCustomRadioDial::SetParams(int APosition, int AMin, int AMax)
{
   bool invalid = false, changed = false;

   // Ensure minimum and maximum in right order.
   if (AMax < AMin)
      throw EInvalidOperation(Consts_SPropertyOutOfRange, ARRAYOFCONST((this->ClassName())));

   // Limit Position to Min and Max.
   if (APosition < AMin) APosition = AMin;
   if (APosition > AMax) APosition = AMax;

   // Change Min if necessary and flag redrawing if so.
   if (FMin != AMin)
   {
      FMin = AMin;
      invalid = true;
   }

   // Change Max if necessary and flag redrawing if so.
   if (FMax != AMax)
   {
      FMax = AMax;
      invalid = true;
   }

   // Change Position if necessary and draw pointer accordingly.
   if (APosition != FPosition)
   {
      FPosition = APosition;
      DrawPointer();
      changed = true;
   }

   // If redrawing flagged, cause a redraw, redoing the bitmap too.
   if (invalid)
   {
      FBitmapInvalid = true;
      changed = true;
      Invalidate();

   }

   if (changed)

      // Notify the user of changes.
      Change();
}


// Set all angle parameters at once.
void __fastcall TCustomRadioDial::SetAngleParams(TRadioAngle AnAngle,
   TRadioAngle AMin, TRadioAngle AMax)
{
   bool invalid = false;

   // Error if AMax < AMin
   if (AMax < AMin)
      throw EInvalidOperation(Consts_SPropertyOutOfRange, ARRAYOFCONST((this->ClassName())));

   // Confine AnAngle to limits.
   if (AnAngle < AMin) AnAngle = AMin;
   if (AnAngle > AMax) AnAngle = AMax;

   // Set MinAngle.
   if (FMinAngle != AMin)
   {
      FMinAngle = AMin;
      invalid = true;
   }

   // Set MaxAngle.
   if (FMaxAngle != AMax)
   {
      FMaxAngle = AMax;
      invalid = true;
   }

   // Redraw if necessary
   if (invalid)
   {
      FBitmapInvalid = true;
      Invalidate();
   }

   // Set Position.
   int pos = AngleToPos(AnAngle);
   if (pos != FPosition)
      SetParams(pos, FMin, FMax);
}


void __fastcall TCustomRadioDial::SetDefaultPos(int Value)
{
   // Change this if side effects are needed, e.g. to show a default pos marker.

   if (Value != FDefaultPos)
      FDefaultPos = Value;
}


void __fastcall TCustomRadioDial::SetFrequency(int Value)
{
   if (Value != FFrequency)
   {
      FFrequency = Value;
      if (FTickStyle == tsAuto)
      {
         ClearTicks();
         SetTicks(FTickStyle);
      }
      FBitmapInvalid = true;
      Invalidate();
   }
}


void __fastcall TCustomRadioDial::SetMin(int Value)
{
   SetParams(FPosition, Value, FMax);
}


void __fastcall TCustomRadioDial::SetMinAngle(TRadioAngle Value)
{
   SetAngleParams(PosToAngle(FPosition), Value, FMaxAngle);
}


void __fastcall TCustomRadioDial::SetMax(int Value)
{
   SetParams(FPosition, FMin, Value);
}


void __fastcall TCustomRadioDial::SetMaxAngle(TRadioAngle Value)
{
   SetAngleParams(PosToAngle(FPosition), FMinAngle, Value);
}


void __fastcall TCustomRadioDial::SetPosition(int Value)
{
   SetParams(Value, FMin, FMax);
}


bool __fastcall TCustomRadioDial::CalcBounds(int &AWidth, int &AHeight)
{
   bool result = false;
   int iSize = rcMinRadius + MinBorder + TickBorder;

   if (FBorderStyle == Radiodial::bsSingle)
      iSize += GetSystemMetrics(SM_CXBORDER);
   iSize *= 2;
   iSize++;
   if (AWidth < iSize)
   {
      AWidth = iSize;
      result = true;
   }
   if (AHeight < iSize)
   {
      AHeight = iSize;
      result = true;
   }
   return result;
}


void __fastcall TCustomRadioDial::SetRadius(int Value)
{
   int maxRadius = (Width <= Height) ? (Width - 1) / 2 : (Height - 1) / 2;

   maxRadius -= (MinBorder + TickBorder);
   if (FBorderStyle == Radiodial::bsSingle)
      maxRadius -= GetSystemMetrics(SM_CXBORDER);
   if (Value > maxRadius) Value = maxRadius;
   if (Value < rcMinRadius) Value = rcMinRadius;
   if (Value != FRadius)
   {
      FRadius = Value;
      FBitmapInvalid = true;
      Invalidate();
   }
   UpdateSize();
}


void __fastcall TCustomRadioDial::SetTicks(TTickStyle Value)
{
   TTickLength len;
   int i;

   if (Value != tsNone)
   {
      SetTick(FMin, tlLong);
      SetTick(FMax, tlLong);
   }
   if (Value == tsAuto)
      for (len = tlMiddle, i = FMin + FFrequency;
           i < FMax;
           len = (len == tlMiddle) ? tlLong : tlMiddle, i += FFrequency)
         SetTick(i, len);
}


void __fastcall TCustomRadioDial::SetTickStyle(TTickStyle Value)
{
   if (FTickStyle != Value)
   {
      FTickStyle = Value;
      ClearTicks();
      SetTicks(Value);
      FBitmapInvalid = true;
      Invalidate();
   }
}


void __fastcall TCustomRadioDial::Change()
{
   if (FOnChange)
      FOnChange(this);
}


void __fastcall TCustomRadioDial::SetSmallChange(int Value)
{
   if (Value >= FLargeChange) Value = FLargeChange / 2;
   if (Value < 1) Value = 1;
   if (Value != FSmallChange) FSmallChange = Value;
}


void __fastcall TCustomRadioDial::SetLargeChange(int Value)
{
   if (Value <= FSmallChange + 1) Value = FSmallChange + 1;
   if (Value != FLargeChange) FLargeChange = Value;
}


void __fastcall TCustomRadioDial::SetTick(int Value, TTickLength Length)
{
   const int lengths[] = {tlShortLen, tlMiddleLen, tlLongLen};
   TTick tick;

   if (Value < FMin || Value > FMax)
      throw EInvalidOperation(Consts_SPropertyOutOfRange, ARRAYOFCONST((this->ClassName())));
   for (unsigned i = 0; i < FTicks.size(); i++)
   {
      tick = FTicks[i];
      if (tick.iValue == Value)
      {
         if (tick.iLength != lengths[Length])
         {
            tick.iLength = lengths[Length];
            tick.bChanged = true;
            Invalidate();
         }
         return;
      }
   }
   TTick newTick(Value, lengths[Length]);
   FTicks.push_back(newTick);
   if (HandleAllocated())
   {
      DrawTick(FBitmap->Canvas, newTick);
      DrawTick(Canvas, newTick);
   }
}


TRadioAngle __fastcall TCustomRadioDial::RadToAngle(const double Radian)
{
   return dRadianToAngle * Radian;
}


double __fastcall TCustomRadioDial::AngleToRad(TRadioAngle AnAngle)
{
   return dAngleToRadian * AnAngle;
}



void __fastcall TCustomRadioDial::DrawTick(TCanvas *ACanvas, TTick Tick)
{
   TPoint point;

   TRadioAngle valueAngle = PosToAngle(Tick.iValue);

   ACanvas->Pen->Color = Tick.cColor;
   point = AngleToPoint(valueAngle, Center, FRadius);
   ACanvas->MoveTo(point.x, point.y);
   point = AngleToPoint(valueAngle, Center, FRadius + Tick.iLength);
   ACanvas->LineTo(point.x, point.y);
   Tick.bChanged = false;
}


void __fastcall TCustomRadioDial::Paint(void)
{
   Canvas->Brush->Color = Parent->Brush->Color;
   Canvas->FillRect(ClientRect);
   BitmapNeeded();
   Canvas->CopyRect(FBitmapRect, FBitmap->Canvas, FBitmapRect);
   DrawBorder();
   DrawPointer();
}


int inline lowest(int a, int b, int c)
{
   return (a < b) ? ((a < c) ? a : c) : ((b < c) ? b : c);
}


int inline highest(int a, int b, int c)
{
   return (a > b) ? ((a > c) ? a : c) : ((b > c) ? b : c);
}

// Purpose: draw pointer on bitmap
// Pre:     clip region set to avoid drawing entire bitmap
// Post:    pointer drawn in given style, shape and color
void __fastcall TCustomRadioDial::DrawPointer(void)
{
   TPoint points[3];
   TPoint& outer = points[0];
   TPoint& inner = points[1];
   TPoint& extra = points[2];
   int innerRadius, dotRadius;
   HRGN region;

   // Don't do anything if no handle yet.
   if (!HandleAllocated()) return;

   // innerRadius is radius of top of disk. ButtonEdge is in percent, so:
   innerRadius = (100 - FButtonEdge) * FRadius / 100 - 1;

   if (FPointerRect.Left < 0) // at startup only
      FPointerRect = Rect(Center.x - innerRadius,
                          Center.y - innerRadius,
                          Center.x + innerRadius + 1,
                          Center.y + innerRadius + 1);

   // Clear old pointer.
   Canvas->CopyRect(FPointerRect, FBitmap->Canvas, FPointerRect);

   Canvas->Pen->Color = FPointerColor;
   Canvas->Brush->Color = FPointerColor;
   switch (FPointerShape)
   {
      case Radiodial::psLine:
         // Draw line.
         outer = AngleToPoint(Angle, Center, innerRadius);
         Canvas->MoveTo(outer.x, outer.y);
         inner = AngleToPoint(Angle, Center, (101 - FPointerSize) * innerRadius / 100);
         Canvas->LineTo(inner.x, inner.y);

         // Set rect for update.
         FPointerRect = Rect(min(inner.x, outer.x),
                             min(inner.y, outer.y),
                             max(inner.x, outer.x),
                             max(inner.y, outer.y));
         break;
      case Radiodial::psTriangle:
         {
            int smallRadius = FPointerSize * innerRadius / 100;

            outer = AngleToPoint(Angle, Center, innerRadius);
            inner = AngleToPoint(TRadioAngle(Angle - 1500), outer, smallRadius);
            extra = AngleToPoint(TRadioAngle(Angle + 1500), outer, smallRadius);
            Canvas->Polygon(points, 2);
            FPointerRect = Rect(lowest(outer.x, inner.x, extra.x),
                                lowest(outer.y, inner.y, extra.y),
                                highest(outer.x, inner.x, extra.x),
                                highest(outer.y, inner.y, extra.y));
         }
         break;
      case Radiodial::psDot:
         dotRadius = FPointerSize * innerRadius / 200;
         outer = AngleToPoint(Angle, Center, innerRadius - dotRadius);
         if (outer.x > Center.x) outer.x++;
         if (outer.y > Center.y) outer.y++;
         FPointerRect = Rect(outer.x - dotRadius, outer.y - dotRadius,
                             outer.x + dotRadius, outer.y + dotRadius);
         Canvas->Ellipse(FPointerRect.Left, FPointerRect.Top,
                         FPointerRect.Right, FPointerRect.Bottom);
         break;
      case Radiodial::psOwnerDraw:
         if (FOnDrawPointer)
         {
            dotRadius = FPointerSize * innerRadius / 200;
            outer = AngleToPoint(Angle, Center, innerRadius - dotRadius);
            if (outer.x > Center.x) outer.x++;
            if (outer.y > Center.y) outer.y++;
            FPointerRect = Rect(outer.x - dotRadius, outer.y - dotRadius,
                                outer.x + dotRadius, outer.y + dotRadius);
            region = CreateEllipticRgn(FPointerRect.Left - 1, FPointerRect.Top - 1,
                                       FPointerRect.Right + 1, FPointerRect.Bottom + 1);
            SelectClipRgn(Canvas->Handle, region);
            Canvas->Pen->Color = PointerColor;
            Canvas->Brush->Color = PointerColor;
            try
            {
               FOnDrawPointer(this, FPointerRect, Canvas);
            }
            catch (...)
            {
               SelectClipRgn(Canvas->Handle, 0);
               DeleteObject(region);
               throw;
            }
         }
         break;
   }
   RECT R(FPointerRect);
   InflateRect(&R, 1, 1);
   FPointerRect = R;
}


void __fastcall TCustomRadioDial::BitmapNeeded(void)
{
   if (!FBitmap)
   {
      FBitmap = new Graphics::TBitmap;
      FBitmapInvalid = true;
   }
   if (FBitmapInvalid)
   {
      if (FBitmap->Width != FSize + 1)
      {
         FBitmap->Width = FSize + 1;
         FBitmap->Height = FSize + 1;
         FBitmapRect = Bounds(0, 0, FSize + 1, FSize + 1);
      }

      // Draw on bitmap.
      DrawButton();
      DrawTicks();
   }
}


TColor __fastcall blendColors(double factor, int first, int second)
{
   double factor2 = 1.0 - factor;
   RGBQUAD *q1 = reinterpret_cast<RGBQUAD *>(&first);
   RGBQUAD *q2 = reinterpret_cast<RGBQUAD *>(&second);
   RGBQUAD result;

   using std::floor;

   result.rgbBlue = floor(factor * q1->rgbBlue + factor2 * q2->rgbBlue);
   result.rgbGreen = floor(factor * q1->rgbGreen + factor2 * q2->rgbGreen);
   result.rgbRed = floor(factor * q1->rgbRed + factor2 * q2->rgbRed);
   result.rgbReserved = 0;
   return *reinterpret_cast<TColor*>(&result);
}


void __fastcall TCustomRadioDial::DrawButton(void)
{
   TPoint oldOrg;
   int size = 2 * FRadius + 1;
   TCanvas *c = FBitmap->Canvas;

   c->Brush->Color = Parent->Brush->Color;
   c->Brush->Style = bsSolid;
   c->FillRect(FBitmapRect);
   SetViewportOrgEx(c->Handle,
                    FSize / 2 - FRadius,
                    FSize / 2 - FRadius,
                    &oldOrg);

   // Draw edge.
   c->Pen->Style = psClear;

   int highlight = ColorToRGB(clBtnHighlight);
   int face = ColorToRGB(this->Color);
   int shadow = (ColorToRGB(Color) & 0x00FEFEFE) >> 1;

   for (int i = 0; i <= size; i++)
   {
      TColor col = blendColors(std::cos(i * M_PI_2 / size), highlight, face);
      c->Brush->Color = col;
      c->Pie(0, 0, size, size, i + 1, 0, i - 1, 0);
      c->Pie(0, 0, size, size, 0, i - 1, 0, i + 1);
   }

   for (int i = 0; i <= size; i++)
   {
      TColor col = blendColors(1.0 - std::sin(i * M_PI_2 / size), face, shadow);
      c->Brush->Color = col;
      c->Pie(0, 0, size, size, size, i + 1, size, i - 1);
      c->Pie(0, 0, size, size, i - 1, size, i + 1, size);
   }

   // Draw top of disk.
   c->Pen->Style = psSolid;
   c->Pen->Color = this->Color;
   c->Brush->Color = this->Color;
   int edge = FButtonEdge * FRadius / 100 + 1;
   c->Ellipse(0 + edge, 0 + edge, 0 + size - edge, 0 + size - edge);

   // Draw bounding circle.
   c->Pen->Color = clBtnText;
   c->Pen->Style = psSolid;
   c->Brush->Style = bsClear;
   c->Ellipse(0, 0, size, size);

   // Reset viewport origin.
   SetViewportOrgEx(c->Handle, oldOrg.x, oldOrg.y, 0);
   FBitmapInvalid = false;
}


void __fastcall TCustomRadioDial::SetPointerShape(TRadioPointerShape Value)
{
   if (Value != FPointerShape)
   {
      FPointerShape = Value;
      Invalidate();
   }
}


void __fastcall TCustomRadioDial::DrawBorder(void)
{
   TRect R = ClientRect;

   Canvas->Brush->Style = bsClear;
   Canvas->Pen->Color = Parent->Brush->Color;
   Canvas->Rectangle(R.Left, R.Top, R.Right, R.Bottom);
   Canvas->Brush->Style = bsSolid;

   if (GetFocus() == Handle)
      Canvas->DrawFocusRect(R);
}


void __fastcall TCustomRadioDial::DrawTicks(void)
{
   if (FTickStyle == tsNone || !FTicks.size())
      return;
   for (unsigned i = 0; i < FTicks.size(); i++)
      DrawTick(FBitmap->Canvas, FTicks[i]);
}


void inline TCustomRadioDial::UpdateSize(void)
{
   FSize = 2 * (MinBorder + FRadius + TickBorder) + 1;
}


void __fastcall TCustomRadioDial::SetBounds(int ALeft, int ATop, int AWidth, int AHeight)
{
   if (CalcBounds(AWidth, AHeight))
     FBitmapInvalid = true;
   inherited::SetBounds(ALeft, ATop, AWidth, AHeight);
   SetRadius(AWidth + AHeight);
}


void __fastcall TCustomRadioDial::CMColorChanged(TMessage &Message)
{
   FBitmapInvalid = true;
   inherited::Dispatch(&Message);
}


void __fastcall TCustomRadioDial::CMParentColorChanged(TMessage &Message)
{
   FBitmapInvalid = true;
   Invalidate();
   inherited::Dispatch(&Message);
}


// Set button edge in percent (0 - 100).
void __fastcall TCustomRadioDial::SetButtonEdge(int Value)
{
  if (Value < rcMinEdge) Value = rcMinEdge;
  if (Value > rcMaxEdge) Value = rcMaxEdge;
  if (Value != FButtonEdge)
  {
     FButtonEdge = Value;
     if (!FBitmapInvalid)
     {
        FBitmapInvalid = true;
        Invalidate();
     }
  }
}


void __fastcall TCustomRadioDial::WMKillFocus(TWMKillFocus &Message)
{
   inherited::Dispatch(&Message);
   if (HandleAllocated())
      DrawBorder();
}


void __fastcall TCustomRadioDial::WMSetFocus(TWMSetFocus &Message)
{
   inherited::Dispatch(&Message);
   if (HandleAllocated())
      DrawBorder();
}


void __fastcall TCustomRadioDial::MouseDown(TMouseButton Button,
   TShiftState Shift, int X, int Y)
{
   inherited::MouseDown(Button, Shift, X, Y);
   if (!Focused())
   {
      SetFocus();
      Invalidate();
   }
   if (PtInRect(reinterpret_cast<RECT*>(&FPointerRect), Point(X, Y)))
      MouseCapture = true;
   else
   {
      TRadioAngle a = RadToAngle(PointToRad(Point(X, Y), Center));
      if (a < Angle)
      {
         DecPos(Shift);
         FIncrementing = false;
      }
      else
      {
         IncPos(Shift);
         FIncrementing = true;
      }
      if (!FRepeatTimer)
         FRepeatTimer = new TTimer(this);
      FRepeatTimer->OnTimer = TimerExpired;
      FRepeatTimer->Interval = FRepeatDelay;
      FRepeatTimer->Enabled = true;
   }
}


void __fastcall TCustomRadioDial::TimerExpired(TObject *Sender)
{
   TShiftState shift = GetShiftState();

   FRepeatTimer->Enabled = false;
   FRepeatTimer->Interval = FRepeatRate;
   if (FIncrementing)
      IncPos(shift);
   else
      DecPos(shift);
   FRepeatTimer->Enabled = true;
}


void __fastcall TCustomRadioDial::MouseMove(TShiftState Shift, int X, int Y)
{
   inherited::MouseMove(Shift, X, Y);
   if (MouseCapture)
      SetAngle(RadToAngle(PointToRad(Point(X, Y), Center)));
}


void __fastcall TCustomRadioDial::MouseUp(TMouseButton Button, TShiftState Shift,
   int X, int Y)
{
   inherited::MouseUp(Button, Shift, X, Y);
   if (FRepeatTimer)
   {
      FRepeatTimer->Enabled = false;
      delete FRepeatTimer;
      FRepeatTimer = 0;
   }
   MouseCapture = false;
}


TPoint inline TCustomRadioDial::GetCenter(void)
{
   return Point(FSize / 2, FSize / 2);
}


void inline TCustomRadioDial::ClearTicks(void)
{
   FTicks.clear();
}


void __fastcall TCustomRadioDial::CreateParams(TCreateParams &Params)
{
   inherited::CreateParams(Params);
   Params.Style |= (FBorderStyle == Radiodial::bsSingle) ? WS_BORDER : 0;
   if (Controls::NewStyleControls &&
       Ctl3D &&
       FBorderStyle == Radiodial::bsSingle)
   {
      Params.Style &= ~WS_BORDER;
      Params.ExStyle |= WS_EX_STATICEDGE;
   }
}


void __fastcall TCustomRadioDial::SetPointerColor(TColor Value)
{
   if (Value != FPointerColor)
   {
      FPointerColor = Value;
      DrawPointer();
   }
}


void __fastcall TCustomRadioDial::CMCtl3DChanged(TMessage &Message)
{
   inherited::Dispatch(&Message);
   FBitmapInvalid = true;
   RecreateWnd();
}


void __fastcall TCustomRadioDial::IncPos(TShiftState Shift)
{
   if (Shift.Contains(ssShift))
      Position = Position + FLargeChange;
   else if (Shift.Contains(ssCtrl))
      Position = FMax;
   else
      Position = Position + FSmallChange;
}


void __fastcall TCustomRadioDial::DecPos(TShiftState Shift)
{
   if (Shift.Contains(ssShift))
      Position = Position - FLargeChange;
   else if (Shift.Contains(ssCtrl))
      Position = FMin;
   else
      Position = Position - FSmallChange;
}


void __fastcall TCustomRadioDial::KeyDown(Word &Key, TShiftState Shift)
{
   switch (Key)
   {
      case VK_UP:
      case VK_RIGHT:
         IncPos(Shift);
         break;
      case VK_DOWN:
      case VK_LEFT:
         DecPos(Shift);
         break;
      case VK_PRIOR: // PgUp
         Shift << ssShift;
         IncPos(Shift);
         break;
      case VK_NEXT: // PgDn
         Shift << ssShift;
         DecPos(Shift);
         break;
      case VK_HOME:
         Position = FMin;
         break;
      case VK_END:
         Position = FMax;
         break;
      default:
         inherited::KeyDown(Key, Shift);
         return;
   }
   Key = 0;
   inherited::KeyDown(Key, Shift);
}


void __fastcall TCustomRadioDial::WndProc(TMessage &Message)
{
   if (Message.Msg == CN_KEYDOWN)
      DoKeyDown(*reinterpret_cast<TWMKey*>(&Message));
   inherited::WndProc(Message);
}


void __fastcall TCustomRadioDial::WMSysColorChange(TWMSysColorChange &Message)
{
   FBitmapInvalid = true;
   Invalidate();
}


void __fastcall TCustomRadioDial::SetPointerSize(int Value)
{
   if (Value > 100) Value = 100;
   else if (Value < 1) Value = 1;
   if (Value != FPointerSize)
   {
      FPointerSize = Value;
      DrawPointer();
   }
}


void __fastcall TCustomRadioDial::Loaded(void)
{
   inherited::Loaded();
   Change();
}

namespace Radiodial
{
   void __fastcall PACKAGE Register()
   {
      TComponentClass classes[1] = {__classid(TRadioDial)};
      RegisterComponents("Samples", classes, 0);
   }
}

