/*
  File:       RadioDial.h
  Function:   TRadioDial component, a button like the dial on a radio.
  Language:   BCB3 (C++)
              Translation and enhancement of the original Object Pascal
              version (by same author)
  Author:     Rudy Velthuis
  Copyright:  (c) 1997,2002 drs. Rudolph Velthuis
  Disclaimer: This code is freeware. All rights are reserved.
              This code is provided as is, expressly without a warranty of
              any kind. You use it at your own risk.
              If you use this code, please credit me.
*/


#ifndef RadioDialH
#define RadioDialH

#include <System.hpp>
#include <SysUtils.hpp>
#include <Controls.hpp>
#include <ComCtrls.hpp>
#include <vector>
#include <cmath>

typedef int TRadioAngle;
typedef Word TRepeatValue;

namespace Radiodial
{
   enum TRadioPointerShape { psLine, psTriangle, psDot, psOwnerDraw };
   enum TTickLength { tlShort, tlMiddle, tlLong };
   enum TRadioBorderStyle { bsNone, bsSingle };
}

class DELPHICLASS TCustomRadioDial;
typedef void __fastcall (__closure *TRadioDrawEvent)
   (TCustomRadioDial* Sender, TRect ARect, TCanvas *ACanvas);

class PACKAGE TCustomRadioDial : public TCustomControl
{
public:
   enum {rcMinEdge = 0, rcMaxEdge = 100, rcMinRadius = 15};
   enum {tlShortLen = 4, tlMiddleLen = 6, tlLongLen = 10};

private:
   typedef TCustomControl inherited;

   const double dRadianToAngle;
   const double dAngleToRadian;

   // Structure containing a tick value. A tick is one of the lines forming
   // the scale of the component.
   // fields
   Graphics::TBitmap *FBitmap;
   TRect FBitmapRect;
   bool FBitmapInvalid;
   Radiodial::TRadioBorderStyle FBorderStyle;
   int FButtonEdge;
   TPoint FCenter;
   int FDefaultPos;
   int FFrequency;
   int FLargeChange;
   int FMax;
   TRadioAngle FMaxAngle;
   int FMin;
   TRadioAngle FMinAngle;
   TRect FPointerRect;
   TColor FPointerColor;
   int FPointerSize;
   Radiodial::TRadioPointerShape FPointerShape;
   int FPosition;
   int FRadius;
   int FSize;
   int FSmallChange;
   TTickStyle FTickStyle;
   bool FIncrementing;
   TTimer *FRepeatTimer;
   TRepeatValue FRepeatRate;
   TRepeatValue FRepeatDelay;

   // events
   TNotifyEvent FOnChange;
   TRadioDrawEvent FOnDrawPointer;
   TScrollEvent FOnScroll;

   bool __fastcall CalcBounds(int &AWidth, int &AHeight);
   TRadioAngle __fastcall GetAngle(void);
   TPoint inline GetCenter(void);
   void __fastcall SetAngle(TRadioAngle Value);
   void __fastcall SetBorderStyle(Radiodial::TRadioBorderStyle Value);
   void __fastcall SetButtonEdge(int Value);
   void __fastcall SetDefaultPos(int Value);
   void __fastcall SetFrequency(int Value);
   void __fastcall SetLargeChange(int Value);
   void __fastcall SetMin(int Value);
   void __fastcall SetMinAngle(TRadioAngle Value);
   void __fastcall SetMax(int Value);
   void __fastcall SetMaxAngle(TRadioAngle Value);
   void __fastcall SetPointerColor(TColor Value);
   void __fastcall SetPointerSize(int Value);
   void __fastcall SetPointerShape(Radiodial::TRadioPointerShape Value);
   void __fastcall SetPosition(int Value);
   void __fastcall SetRadius(int Value);
   void __fastcall SetSmallChange(int Value);
   void __fastcall SetTickStyle(TTickStyle Value);
   void inline UpdateSize(void);
   void __fastcall TimerExpired(TObject *Sender);

protected:
   struct TTick
   {
      int iValue;          // Position
      int iLength;         // Length of line
      TColor cColor;       // Color of line
      bool bChanged;       // True if must be redrawn

      TTick(int Value, int Length = tlMiddleLen) :
         iValue(Value), iLength(Length), cColor(clBtnText), bChanged(true) { }
      TTick() :
         iValue(0), iLength(tlMiddleLen), cColor(clBtnText), bChanged(true) { }
   };

   typedef std::vector<TTick> TTickVect;

   TTickVect FTicks;

   int __fastcall AngleToPos(TRadioAngle AnAngle);
   DYNAMIC void __fastcall BitmapNeeded(void);
   DYNAMIC void __fastcall Change(void);
   void inline ClearTicks(void);
   MESSAGE void __fastcall CMColorChanged(TMessage &Message);
   MESSAGE void __fastcall CMParentColorChanged(TMessage &Message);
   virtual void __fastcall CreateParams(Controls::TCreateParams &Params);
   DYNAMIC void __fastcall DrawBorder(void);
   DYNAMIC void __fastcall DrawButton(void);
   DYNAMIC void __fastcall DrawPointer(void);
   DYNAMIC void __fastcall DrawTick(TCanvas *ACanvas, TTick Tick);
   DYNAMIC void __fastcall DrawTicks(void);
   DYNAMIC void __fastcall KeyDown(Word &Key, Classes::TShiftState Shift);
   virtual void __fastcall Loaded(void);
   DYNAMIC void __fastcall MouseDown(TMouseButton Button, Classes::TShiftState Shift, int X, int Y);
   DYNAMIC void __fastcall MouseMove(Classes::TShiftState Shift, int X, int Y);
   DYNAMIC void __fastcall MouseUp(TMouseButton Button, Classes::TShiftState Shift, int X, int Y);
   virtual void __fastcall Paint(void);
   TRadioAngle __fastcall PosToAngle(int Pos);
   virtual void __fastcall SetTicks(TTickStyle Value);
   MESSAGE void __fastcall CMCtl3DChanged(TMessage &Message);
   MESSAGE void __fastcall WMSetFocus(TWMSetFocus &Message);
   MESSAGE void __fastcall WMKillFocus(TWMKillFocus &Message);
   MESSAGE void __fastcall WMSysColorChange(TWMSysColorChange &Message);
   virtual void __fastcall WndProc(TMessage &Message);
   DYNAMIC void __fastcall DecPos(TShiftState Shift);
   DYNAMIC void __fastcall IncPos(TShiftState Shift);
   __property TTickVect Ticks = {read=FTicks, write=FTicks, stored=true};

   // properties to be published;
   __property TRadioAngle Angle = {read=GetAngle, write=SetAngle, stored=false, nodefault};
   __property Radiodial::TRadioBorderStyle BorderStyle = {read=FBorderStyle, write=SetBorderStyle, default=Radiodial::bsNone};
   __property int ButtonEdge = {read=FButtonEdge, write=SetButtonEdge, default=5};
   __property int DefaultPos = {read=FDefaultPos, write=SetDefaultPos};
   __property int Frequency = {read=FFrequency, write=SetFrequency, default=10};
   __property int LargeChange = {read=FLargeChange, write=SetLargeChange, default=2};
   __property int Max = {read=FMax, write=SetMax, default=100};
   __property TRadioAngle MaxAngle = {read=FMaxAngle, write=SetMaxAngle, default=3300};
   __property int Min = {read=FMin, write=SetMin, default=0};
   __property TRadioAngle MinAngle = {read=FMinAngle, write=SetMinAngle, default=300};
   __property TColor PointerColor = {read=FPointerColor, write=SetPointerColor, default=clBtnText};
   __property int PointerSize = {read=FPointerSize, write=SetPointerSize, default=33};
   __property Radiodial::TRadioPointerShape PointerShape = {read=FPointerShape, write=SetPointerShape, default=Radiodial::psLine};
   __property int Position = {read=FPosition, write=SetPosition, default=0};
   __property int Radius = {read=FRadius, write=SetRadius, nodefault};
   __property TRepeatValue RepeatDelay = {read=FRepeatDelay, write=FRepeatDelay, default=400};
   __property TRepeatValue RepeatRate = {read=FRepeatRate, write=FRepeatRate, default=100};
   __property int SmallChange = {read=FSmallChange, write=SetSmallChange, default=1};
   __property TTickStyle TickStyle = {read=FTickStyle, write=SetTickStyle, stored=true};
   __property TabStop = {default=true};

   // events
   __property TNotifyEvent OnChange = {read=FOnChange, write=FOnChange};
   __property TRadioDrawEvent OnDrawPointer = {read=FOnDrawPointer, write=FOnDrawPointer};

public:
   __fastcall virtual TCustomRadioDial(TComponent* Owner);
   __fastcall virtual ~TCustomRadioDial(void);

   TPoint __fastcall AngleToPoint(TRadioAngle AnAngle, const TPoint &ACenter, int ARadius);
   virtual void __fastcall SetAngleParams(TRadioAngle AnAngle, TRadioAngle AMin, TRadioAngle AMax);
   virtual void __fastcall SetBounds(int ALeft, int ATop, int AWidth, int AHeight);
   virtual void __fastcall SetParams(int APosition, int AMin, int AMax);
   virtual void __fastcall SetTick(int Value, Radiodial::TTickLength Length);
   TRadioAngle __fastcall RadToAngle(const double Radian);
   double __fastcall AngleToRad(TRadioAngle AnAngle);

   __property Graphics::TBitmap *Bitmap = {read=FBitmap};
   __property Windows::TPoint Center = {read=GetCenter};

__published:

BEGIN_MESSAGE_MAP
   VCL_MESSAGE_HANDLER(CM_PARENTCOLORCHANGED, TMessage, CMParentColorChanged);
   VCL_MESSAGE_HANDLER(CM_COLORCHANGED, TMessage, CMColorChanged);
   VCL_MESSAGE_HANDLER(CM_CTL3DCHANGED, TMessage, CMCtl3DChanged);
   VCL_MESSAGE_HANDLER(WM_SETFOCUS, TWMSetFocus, WMSetFocus);
   VCL_MESSAGE_HANDLER(WM_KILLFOCUS, TWMKillFocus, WMKillFocus);
   VCL_MESSAGE_HANDLER(WM_SYSCOLORCHANGE, TWMSysColorChange, WMSysColorChange);
END_MESSAGE_MAP(TCustomControl);
};


class PACKAGE TRadioDial : public TCustomRadioDial
{
__published:

   // properties
   __property Align;
   __property Angle;
   __property BorderStyle;
   __property ButtonEdge;
   __property Color;
   __property Ctl3D;
   __property Cursor;
   __property DefaultPos;
   __property DragCursor;
   __property DragMode;
   __property Enabled;
   __property Frequency;
   __property LargeChange;
   __property Max;
   __property MaxAngle;
   __property Min;
   __property MinAngle;
   __property ParentColor;
   __property ParentCtl3D;
   __property ParentShowHint;
   __property PointerColor;
   __property PointerSize;
   __property PointerShape;
   __property PopupMenu;
   __property Position;
   __property Radius;
   __property RepeatDelay;
   __property RepeatRate;
   __property ShowHint;
   __property SmallChange;
   __property TickStyle;
   __property TabOrder;
   __property TabStop;
   __property Visible;

   // events
   __property OnChange;
   __property OnClick;
   __property OnDblClick;
   __property OnDragDrop;
   __property OnDragOver;
   __property OnDrawPointer;
   __property OnEndDrag;
   __property OnEnter;
   __property OnExit;
   __property OnKeyDown;
   __property OnKeyPress;
   __property OnKeyUp;
   __property OnMouseDown;
   __property OnMouseMove;
   __property OnMouseUp;
   __property OnStartDrag;

// BCB4 and higher (untested)   
#ifdef __BORLANDC__ >= 0x540
   __property Anchors;
   __property Constraints;

   __property OnEndDock;
   __property OnStartDock;
#endif

};

#if !defined(NO_IMPLICIT_NAMESPACE_USE)
using namespace Radiodial;
#endif

#endif
