/////////////////////////////////////////////////////////////////////////////
// File:       RadioDial.cs                                                //
// Function:   RadioDial component, a button like the dial on a radio.     //
//             This is a C# adaptation of my Delphi RadioDial control.     //
// Language:   C#                                                          //
// Author:     Rudolph Velthuis                                            //
// Copyright:  (c) 1997,2003 drs. Rudolph Velthuis                         //
// Disclaimer: This code is freeware. All rights are reserved.             //
//             This code is provided as is, expressly without a warranty   //
//             of any kind. You use it at your own risk.                   //
//                                                                         //
//             If you use this code, please credit me.                     //
//                                                                         //
// To do:      - AutoSize                                                  //
//             - more pointer types                                        //
//             - add descriptions to properties                            //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

using System;
using System.ComponentModel;
using System.Collections;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using Velthuis.Win32;

namespace Velthuis.CustCtrls
{
    // Tick styles for RadioControl.
    public enum TickStyle
    {
        None,
        Auto,
        Manual
    }

    // Pointer shapes for RadioControl.
    public enum PointerShape
    {
        Line,
        Triangle,
        Dot,
        OwnerDraw
    }

	/// <summary>
	/// RadioDial component, a button like the dial on a radio.
	/// </summary>
	public class RadioDial : Control
	{
        // constants useful for converting angles to radians and back.
        public const double dAngleToRadian = Math.PI / 1800;
        public const double dRadianToAngle = 1800 / Math.PI;

        // Size of edges, in percent
        public const int MinEdge = 0;
        public const int MaxEdge = 100;

        // the minimum radius a radio dial should have
        protected const int MinRadius = 15;

        // Lengths for short, middle and long tick
        public enum TickLength
        {
            Long      = 10,
            Middle    = 6,
            Short     = 4
        }

        // Arguments given to a DrawPointer handler, if the radio dial
        // has an owner drawn pointer.
        public class RadioDrawEventArgs
        {
            private Rectangle mRect;
            private Graphics mGraphics;

            public RadioDrawEventArgs(Rectangle rect, Graphics graphics)
            {
                mRect = rect;
                mGraphics = graphics;
            }
            public Rectangle DrawRect { get { return mRect; } }
            public Graphics Graphics { get { return mGraphics; } }
        }

        // Event type of the event when an owner drawn pointer is required.
        public delegate void RadioDrawEventHandler(RadioDial sender,
            RadioDrawEventArgs e);

        // The struct that represents one tick on the perimeter of the
        // dial.
        public struct Tick
        {
            public int Value;
            public TickLength Length;
            public Color Color;
            public bool Changed;
            public Tick(bool changed)
            {
                Changed = changed;
                Value = 0;
                Length = TickLength.Short;
                Color = Color.Empty;
            }
            public Tick(int value, TickLength length, Color color, bool changed)
            {
                Value = value;
                Length = length;
                Color = color;
                Changed = changed;
            }
        }

        #region private data members

        // Minimum border for the BorderStyle border.
        private const int cMinBorder = 1;
        // Mimimum border to make room for the ticks.
        private const int cTickBorder = (int)TickLength.Long;

        private Bitmap mBitmap;
        private bool mBitmapInvalid;
        private BorderStyle mBorderStyle;
        private int mButtonEdge;
        private int mDefaultValue;
        private int mTickFrequency;
        private int mLargeChange;
        private int mMaximum;
        private int mMaximumAngle;
        private int mMinimum;
        private int mMinimumAngle;
        private Rectangle mPointerRect;
        private Color mPointerColor;
        private int mPointerSize;
        private PointerShape mPointerShape;
        private int mValue;
        private int mRadius;
        private int mSize;
        private int mSmallChange;
        private ArrayList mTicks;
        private TickStyle mTickStyle;
        private bool mIncrementing;
        private System.Timers.Timer mRepeatTimer;
        private int mRepeatRate;
        private int mRepeatDelay;
        private bool mHandleAllocated = false;
        #endregion

        #region private methods

        // Calculate the minimum size of the control.
        private bool CalcBounds(ref int width, ref int height)
        {
            int size;
            bool result = false;

            size = MinRadius + cMinBorder + (int)TickLength.Long;
            if (mBorderStyle == BorderStyle.FixedSingle)
                size += SystemMetrics.CXBorder;
            size = 2 * size + 1;
            if (width < size)
            {
                width = size;
                result = true;
            }
            if (height < size)
            {
                height = size;
                result = true;
            }
            return result;
        }

        // Convert "point" to an angle (relative to "center") in radians,
        // where bottom is 0, left is Pi/2, top is Pi and so on.
        private double PointToRad(Point point, Point center)
        {
            double result;

            int n = point.X - center.X;
            if (n == 0)
                result = 0.5 * Math.PI;
            else
                result = Math.Atan((double)(center.Y - point.Y) / n);
            if (n < 0)
                result += Math.PI;
            return 1.5 * Math.PI - result;
        }

        // Makes sure the internal size is up to date.
        private void UpdateSize()
        {
             mSize = 2 * (cMinBorder + mRadius + (int)TickLength.Long) + 1;
        }

        // Handle timer elapsed events for mouse repeats.
        private void TimerExpired(object source, System.Timers.ElapsedEventArgs e)
        {
             mRepeatTimer.Enabled = false;
             mRepeatTimer.Interval = mRepeatRate;
             if (mIncrementing)
                 IncPos(Control.ModifierKeys);
             else
                 DecPos(Control.ModifierKeys);
             mRepeatTimer.Enabled = true;
        }
        #endregion

        #region protected methods
        // Make sure that arrow keys (with or without Shift) are used for
        // input, and not for form navigation.
        protected override bool IsInputKey(Keys keyData)
        {
            switch (keyData & ~Keys.Shift)
            {
                case Keys.Up:
                case Keys.Down:
                case Keys.Left:
                case Keys.Right:
                    return true;
            }
            return base.IsInputKey(keyData);
        }

        // Remember if a control has a handle or not.
        protected override void OnHandleCreated(EventArgs e)
        {
            base.OnHandleCreated(e);
            mHandleAllocated = true;
        }

        // Remember if a control has a handle or not.
        protected override void OnHandleDestroyed(EventArgs e)
        {
            base.OnHandleDestroyed(e);
            mHandleAllocated = false;
        }

        // React on changes of the back color
        protected override void OnBackColorChanged(EventArgs e)
        {
            base.OnBackColorChanged(e);
            mBitmapInvalid = true;
            Invalidate();
        }


        // Converts an Angle value to a Value value.
        protected int AngleToValue(int angle)
        {
            return mMinimum + (mMaximum - mMinimum) *
                              (angle - mMinimumAngle) /
                              (mMaximumAngle - mMinimumAngle);
        }

        // Ensures a bitmap is there when we need it.
        // The bitmap is used for the internal buffering of the dial.
        protected virtual void BitmapNeeded()
        {
            if (mBitmap == null)
            {
                mBitmap = new Bitmap(mSize + 1, mSize + 1);
                mBitmapInvalid = true;
            }
            if (mBitmapInvalid)
            {
                if (mBitmap.Width != mSize + 1)
                {
                    mBitmap = new Bitmap(mSize + 1, mSize + 1);
                }
                DrawButton();
                DrawTicks();
            }
        }

        // Handles changes of Value by calling ValueChanged
        protected virtual void OnValueChanged()
        {
            if (ValueChanged != null)
                ValueChanged(this, EventArgs.Empty);
        }

        // Clears all ticks.
        protected void ClearTicks()
        {
            mTicks.Clear();
        }

        private const int WS_BORDER = 0x800000;
        private const int WS_EX_STATICEDGE = 0x20000;

        // Set border styles according to BorderStyle property
        protected override CreateParams CreateParams
        {

            get
            {
                CreateParams old = base.CreateParams;
                switch (mBorderStyle)
                {
                    case BorderStyle.FixedSingle:
                        old.Style = old.Style | WS_BORDER;
                        break;
                    case BorderStyle.Fixed3D:
                        old.ExStyle = old.ExStyle | WS_EX_STATICEDGE;
                        break;
                }
                return old;
            }
        }


        // Draw border, mainly to display the focus rectangle.
        protected virtual void DrawFocusRectangle()
        {
            // Couldn't get PaintControl.DrawFocusRectangle to work as I
            // wanted, so I had to do this.

            Rectangle rect = ClientRectangle;
            rect.Width -= 1;
            rect.Height -= 1;
            rect.Inflate(-1, -1);

            Graphics canvas = Graphics.FromHwnd(Handle);
            Pen pen;

            if (this.Focused)
            {
                pen = new Pen(Color.FromKnownColor(KnownColor.ControlText));
                pen.DashStyle = DashStyle.Dot;
            }
            else
            {
                pen = new Pen(BackColor);
                pen.DashStyle = DashStyle.Solid;
            }
            canvas.DrawRectangle(pen, rect);
            pen.Dispose();
            canvas.Dispose();
        }

        // Draw button (on bitmap).
        protected virtual void DrawButton()
        {
            int size = 2 * mRadius + 1;
            Rectangle buttonRect = new Rectangle(0, 0, size, size);

            // Clear background.
            SolidBrush brush = new SolidBrush(BackColor);
            Rectangle bitmapRect = new Rectangle(0, 0, mSize, mSize);

            // defer this to the latest possible point in time!
            Graphics c = Graphics.FromImage(mBitmap);
            c.FillRectangle(brush, bitmapRect);
            brush.Dispose();

            // Set drawing origin to top left of circle
            c.TranslateTransform(mSize / 2 - mRadius, mSize / 2 - mRadius);

            // Points for GraphicsPath.
            Point[] points =
            {
                new Point(0, 0),
                new Point(size, 0),
                new Point(size, size),
                new Point(0,size)
            };
            GraphicsPath path = new GraphicsPath();
            path.AddLines(points);

            // Draw sides as one gradient disc.
            Color highlight = Color.FromKnownColor(KnownColor.ControlLightLight);
            Color face = ForeColor;
            Color shadow = Color.FromArgb(ForeColor.R / 2, ForeColor.G / 2, ForeColor.B / 2);
            PathGradientBrush gradient = new PathGradientBrush(path);
            gradient.CenterColor = ForeColor;
            gradient.SurroundColors =
                new Color[] { highlight, face, shadow, face };

            c.FillEllipse(gradient, 0, 0, size, size);
            gradient.Dispose();

            // Draw top
            brush = new SolidBrush(face);
            int edge = mButtonEdge * mRadius / 100 + 1;

            // Don't forget that also Ellipse is drawn (x, y, width, height)
            c.FillEllipse(brush, edge, edge, size - 2 * edge, size - 2 * edge);
            brush.Dispose();

            Pen pen = new Pen(Color.FromKnownColor(KnownColor.ControlText));
            c.SmoothingMode = SmoothingMode.AntiAlias;
            c.DrawEllipse(pen, 0, 0, size, size);

            c.Dispose();

            mBitmapInvalid = false;
        }

        private int lowest(int a, int b, int c)
        {
            return Math.Min(a, Math.Min(b, c));
        }

        private int highest(int a, int b, int c)
        {
            return Math.Max(a, Math.Max(b, c));
        }

        protected virtual void OnDrawPointer()
        {
            if (!mHandleAllocated)
                return;

            int innerRadius = (100 - mButtonEdge) * mRadius / 100 - 1;
            if (mPointerRect.Left < 0)
                mPointerRect = new Rectangle(
                    Center.X - innerRadius,
                    Center.Y - innerRadius,
                    2 * innerRadius + 1,  // width!
                    2 * innerRadius + 1); // height!

            // Be sure to dispose of this one!
            Graphics canvas = Graphics.FromHwnd(Handle);
            canvas.DrawImage(mBitmap, mPointerRect, mPointerRect,
                GraphicsUnit.Pixel);
            // This is for a solid dot. I'd also like to make a Ctl3D type of
            // dot or an open type of dot. I'd also have to make a disabled
            // type of dot.
            Pen pen = new Pen(mPointerColor);
            Brush brush = new SolidBrush(mPointerColor);

            try
            {
                Point inner, outer, extra;

                switch (mPointerShape)
                {
                    case PointerShape.Line:
                        outer = AngleToPoint(Angle, Center, innerRadius);
                        inner = AngleToPoint(Angle, Center, (101 - mPointerSize) * innerRadius / 100);

                        canvas.DrawLine(pen, outer, inner);

                        mPointerRect = new Rectangle(
                            Math.Min(inner.X, outer.X),
                            Math.Min(inner.Y, outer.Y),
                            Math.Abs(inner.X - outer.X),  // width!
                            Math.Abs(inner.Y - outer.Y)); // height!
                        break;
                    case PointerShape.Triangle:
                        int smallRadius = mPointerSize * innerRadius / 100;
                        outer = AngleToPoint(Angle, Center, innerRadius);
                        inner = AngleToPoint(Angle - 1500, outer, smallRadius);
                        extra = AngleToPoint(Angle + 1500, outer, smallRadius);

                        canvas.FillPolygon(brush, new Point[]
                            { outer, inner, extra });
                        int left = lowest(outer.X, inner.X, extra.X);
                        int top = lowest(outer.Y, inner.Y, extra.Y);

                        mPointerRect = new Rectangle(
                           left, top,
                           highest(outer.X, inner.X, extra.X) - left,
                           highest(outer.Y, inner.Y, extra.Y) - top);
                        break;
                    case PointerShape.Dot:
                        int dotRadius = mPointerSize * innerRadius / 200;
                        inner = AngleToPoint(Angle, Center,
                            innerRadius - dotRadius);
                        if (inner.X > Center.X) inner.X++;
                        if (inner.Y > Center.Y) inner.Y++;
                        mPointerRect = new Rectangle(
                            inner.X - dotRadius,
                            inner.Y - dotRadius,
                            2 * dotRadius + 1,
                            2 * dotRadius + 1);
                        canvas.FillEllipse(brush, mPointerRect);
                        break;
                    case PointerShape.OwnerDraw:
                        if (DrawPointer != null)
                        {
                            dotRadius = mPointerSize * innerRadius / 200;
                            outer = AngleToPoint(Angle, Center, innerRadius - dotRadius);
                            if (outer.X > Center.X) outer.X++;
                            if (outer.Y > Center.Y) outer.Y++;
                            mPointerRect = new Rectangle(
                               outer.X - dotRadius, outer.Y - dotRadius,
                               2 * dotRadius, 2 * dotRadius);
                            // Create a clipping region to protect the area
                            // outside the button face.

                            GraphicsPath ellipsePath = new GraphicsPath();
                            mPointerRect.Inflate(1, 1);
                            ellipsePath.AddEllipse(mPointerRect);
                            mPointerRect.Inflate(-1, -1);
                            canvas.Clip = new Region(ellipsePath);

                            try
                            {
                                DrawPointer(this,
                                    new RadioDrawEventArgs(mPointerRect, canvas));
                            }
                            finally
                            {
                                ellipsePath.Dispose();
                                canvas.Clip.Dispose();
                            }
                        }
                        break;
                }
            }
            finally
            {
                mPointerRect.Inflate(1, 1);
                pen.Dispose();
                brush.Dispose();
                canvas.Dispose();
            }
        }

        protected virtual void DrawTick(Graphics canvas, ref Tick T)
        {
            int valueAngle = PosToAngle(T.Value);
            Pen pen = new Pen(T.Color);
            Point p1 = AngleToPoint(valueAngle, Center, mRadius);
            Point p2 = AngleToPoint(valueAngle, Center, mRadius + (int)T.Length);
            canvas.DrawLine(pen, p1, p2);
            pen.Dispose();
            T.Changed = false;
        }

        protected virtual void DrawTicks()
        {
            Tick t;
            if ((mTickStyle != TickStyle.None) &&
                (mTicks != null) &&
                (mTicks.Count != 0))
            foreach (object obj in mTicks)
            {
                t = (Tick)obj;
                if (mBitmap != null)
                    DrawTick(Graphics.FromImage(mBitmap), ref t);
            }
        }

        protected override void OnKeyDown(KeyEventArgs e)
        {
            switch (e.KeyCode)
            {
                case Keys.Up:
                case Keys.Right:
                    IncPos(e.Modifiers);
                    break;
                case Keys.Down:
                case Keys.Left:
                    DecPos(e.Modifiers);
                    break;
                case Keys.PageUp:
                    IncPos(e.Modifiers | Keys.Shift);
                    break;
                case Keys.PageDown:
                    DecPos(e.Modifiers | Keys.Shift);
                    break;
                case Keys.Home:
                    Value = mMinimum;
                    break;
                case Keys.End:
                    Value = mMaximum;
                    break;
                default:
                    base.OnKeyDown(e);
                    return;
            }
            e.Handled = true;

//            base.OnKeyDown(e);
        }

        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);
            if (!Focused)
            {
                Focus();
                Invalidate();
            }
            if (mPointerRect.Contains(e.X, e.Y))
                Capture = true;
            else
            {
                int a = RadToAngle(PointToRad(new Point(e.X, e.Y), Center));
                if (a < Angle)
                {
                    DecPos(Control.ModifierKeys);
                    mIncrementing = false;
                }
                else
                {
                    IncPos(Control.ModifierKeys);
                    mIncrementing = true;
                }
                if (mRepeatTimer == null)
                    mRepeatTimer = new System.Timers.Timer();
                mRepeatTimer.Interval = mRepeatDelay;
                mRepeatTimer.Elapsed += new System.Timers.ElapsedEventHandler(TimerExpired);
                mRepeatTimer.Enabled = true;
            }
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);
            if (Capture)
                Angle = RadToAngle(PointToRad(new Point(e.X, e.Y), Center));
        }

        protected override void OnMouseUp(MouseEventArgs e)
        {
            base.OnMouseUp(e);
            if (mRepeatTimer != null)
                mRepeatTimer.Enabled = false;
            Capture = false;
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            SolidBrush brush = new SolidBrush(Parent.BackColor);
            e.Graphics.FillRectangle(brush, DisplayRectangle);
            UpdateSize();
            BitmapNeeded();
            e.Graphics.DrawImage(mBitmap, 0, 0);
            DrawFocusRectangle();
            OnDrawPointer();
        }

        protected int PosToAngle(int pos)
        {
            return mMinimumAngle + ((mMaximumAngle - mMinimumAngle) *
                                (pos - mMinimum) / (mMaximum - mMinimum));
        }

        protected virtual void SetTicks(TickStyle value)
        {
            mTicks.Clear();
            if (value != TickStyle.None)
            {
                SetTick(mMinimum, TickLength.Long);
                SetTick(mMaximum, TickLength.Long);
            }
            if (value == TickStyle.Auto)
            {
                TickLength len = TickLength.Middle;
                for (int i = mMinimum + mTickFrequency; i < mMaximum; i += mTickFrequency)
                {
                    SetTick(i, len);
                    if (len == TickLength.Middle)
                        len = TickLength.Long;
                    else
                        len = TickLength.Middle;
                }
            }
        }

        protected virtual void IncPos(Keys shift)
        {
            if ((shift & Keys.Shift) == Keys.Shift)
                Value += mLargeChange;
            else if ((shift & Keys.Control) == Keys.Control)
                Value = mMaximum;
            else
                Value += mSmallChange;
        }

        protected virtual void DecPos(Keys shift)
        {
            if ((shift & Keys.Shift) == Keys.Shift)
                Value -= mLargeChange;
            else if ((shift & Keys.Control) == Keys.Control)
                Value = mMinimum;
            else
                Value -= mSmallChange;
        }

        protected override void OnForeColorChanged(EventArgs e)
        {
            mBitmapInvalid = true;
            base.OnForeColorChanged(e);
        }

        protected override void OnParentBackColorChanged(EventArgs e)
        {
            mBitmapInvalid = true;
            base.OnParentBackColorChanged(e);
        }

        protected override void OnGotFocus(EventArgs e)
        {
            base.OnGotFocus(e);
            if (mHandleAllocated)
                DrawFocusRectangle();
        }

        protected override void OnLostFocus(EventArgs e)
        {
            base.OnLostFocus(e);
            if (mHandleAllocated)
                DrawFocusRectangle();
        }

        protected override void OnSystemColorsChanged(EventArgs e)
        {
            mBitmapInvalid = true;
            Invalidate();
        }

		protected override void Dispose(bool disposing)
		{
			base.Dispose(disposing);
		}

        protected override void SetBoundsCore(int x, int y, int width,
            int height, BoundsSpecified specified)
        {
            if (CalcBounds(ref width, ref height))
                mBitmapInvalid = true;
            base.SetBoundsCore(x, y, width, height, specified);
            Radius = width + height;
        }

        /*
        protected override void WndProc(ref Message m)
        {
        }
        */
        #endregion

        #region public methods
		public RadioDial() : base()
		{
            SetStyle(ControlStyles.StandardClick | ControlStyles.UserPaint |
                     ControlStyles.Selectable, true);
            SetStyle(ControlStyles.DoubleBuffer, false);
            UpdateStyles();
            mTicks = new ArrayList();
            mBorderStyle = BorderStyle.None;
            mButtonEdge = 5;
            mDefaultValue = 0;
            mTickFrequency = 1;
            mLargeChange = 2;
            mMaximum = 10;
            mMaximumAngle = 3300;
            mMinimum = 0;
            mMinimumAngle = 300;
            mPointerColor = Color.FromKnownColor(KnownColor.ControlText);
            mPointerSize = 33;
            mRadius = MinRadius;
            mSmallChange = 1;
            TabStop = true;
            mTickStyle = TickStyle.Auto;
            mBitmapInvalid = true;
            mPointerRect = new Rectangle(-1, 0, 0, 0);
            Width = 51;
            Height = 51;
            mRepeatDelay = 400;
            mRepeatRate = 100;
            Value = 0;
            ForeColor = Color.FromKnownColor(KnownColor.Control);
            SetTicks(mTickStyle);
		}

        public Point AngleToPoint(int angle, Point center, int radius)
        {
            double radAngle = AngleToRad(angle);
            return new Point(
                center.X - (int)Math.Round(radius * Math.Sin(radAngle)),
                center.Y + (int)Math.Round(radius * Math.Cos(radAngle)));
        }

        public virtual void SetAngleParams(int angle, int min, int max)
        {
            if (max < min)
                throw new Exception("Property out of range");

            if (angle < min)
                angle = min;
            else if (angle > max)
                angle = max;

            bool invalid = false;

            if (mMinimumAngle != min)
            {
                mMinimumAngle = min;
                invalid = true;
            }

            if (mMaximumAngle != max)
            {
                mMaximumAngle = max;
                invalid = true;
            }

            if (invalid)
            {
                mBitmapInvalid = true;
                Invalidate();
            }

            int pos = AngleToValue(angle);
            if (pos != mValue)
                SetParams(pos, mMinimum, mMaximum);
        }

        public virtual void SetParams(int pos, int min, int max)
        {

            if (max < min)
                throw new Exception("Property out of range");

            if (pos < min)
                pos = min;
            else if (pos > max)
                pos = max;

            bool changed = false;
            bool invalid = false;
            bool drawTicks = false;

            if (mMinimum != min)
            {
                mMinimum = min;
                invalid = true;
                drawTicks= true;
            }

            if (mMaximum != max)
            {
                mMaximum = max;
                invalid = true;
                drawTicks = true;
            }

            if (mValue != pos)
            {
                mValue = pos;
                OnDrawPointer();
                changed = true;
            }

            if (drawTicks)
            {
                SetTicks(mTickStyle);
            }

            if (invalid)
            {
                mBitmapInvalid = true;
                Invalidate();
            }

            if (changed)
                OnValueChanged();
        }


        public virtual void SetTick(int value, TickLength length)
        {
            if ((value < mMinimum) || (value > mMaximum))
                throw new Exception("Tick value out of range");
            for (int i = 0; i < mTicks.Count; ++i)
            {
                Tick t = (Tick)mTicks[i];
                if (t.Value == value)
                {
                    if (t.Length != length)
                    {
                        t.Length = length;
                        t.Changed = true;
                        Invalidate();
                    }
                    return;
                }
            }
            Tick n = new Tick(value, length,
                Color.FromKnownColor(KnownColor.ControlText), true);
            mTicks.Add(n);
            if (mHandleAllocated)
            {
                Graphics canvas = Graphics.FromImage(mBitmap);
                DrawTick(canvas, ref n);
                canvas.Dispose();
                canvas = Graphics.FromHwnd(Handle);
                DrawTick(canvas, ref n);
                canvas.Dispose();
            }
        }

        public int RadToAngle(double radian)
        {
            return (int)Math.Round(radian * dRadianToAngle);
        }

        public double AngleToRad(int angle)
        {
            return angle * dAngleToRadian;
        }
        #endregion

        #region Properties

        [Category("Appearance")]
        public BorderStyle BorderStyle
        {
            get { return mBorderStyle; }
            set
            {
                if (value != mBorderStyle)
                {
                    mBorderStyle = value;
                    if (mHandleAllocated)
                    {
                        RecreateHandle();
                        DrawFocusRectangle();
                    }
                }
            }
        }

        [Category("Appearance")]
        public int ButtonEdge
        {
            get { return mButtonEdge; }
            set
            {
                if (value < MinEdge)
                    value = MinEdge;
                else if (value > MaxEdge)
                    value = MaxEdge;
                if (value != mButtonEdge)
                {
                    mButtonEdge = value;
                    if (!mBitmapInvalid)
                    {
                        mBitmapInvalid = true;
                        Invalidate();
                    }
                }
            }
        }

        [
            Category("Appearance"),
            Description("The number of positions between tick marks.")
        ]
        public int TickFrequency
        {
            get { return mTickFrequency; }
            set
            {
                if (value != mTickFrequency)
                {
                    mTickFrequency = value;
                    if (mTickStyle == TickStyle.Auto)
                    {
                        ClearTicks();
                        SetTicks(mTickStyle);
                    }
                    mBitmapInvalid = true;
                    Invalidate();
                }
            }
        }

        [Category("Appearance")]
        public Color PointerColor
        {
            get { return mPointerColor; }
            set
            {
                if (value != mPointerColor)
                {
                    mPointerColor = value;
                    OnDrawPointer();
                }
            }
        }

        [Category("Appearance")]
        public int PointerSize
        {
            get { return mPointerSize; }
            set
            {
                if (value > 100)
                    value = 100;
                else if (value < 1)
                    value = 1;
                if (value != mPointerSize)
                {
                    mPointerSize = value;
                    OnDrawPointer();
                }
            }
        }

        [
            Category("Appearance"),
            Description("The shape of the pointer, or OwnerDraw.")
        ]
        public PointerShape PointerShape
        {
            get { return mPointerShape; }
            set
            {
                if (value != mPointerShape)
                {
                    mPointerShape = value;
                    Invalidate();
                }
            }
        }

        [
            Category("Appearance"),
            Description("The radius of the dial.")
        ]
        public int Radius
        {
            get { return mRadius; }
            set
            {
                int maxRadius;

                if (Width <= Height)
                    maxRadius = (Width - 1) / 2 - cMinBorder - cTickBorder;
                else
                    maxRadius = (Height - 1) / 2 - cMinBorder - cTickBorder;
                if (mBorderStyle == BorderStyle.FixedSingle)
                    maxRadius -= SystemMetrics.CXBorder;

                if (value > maxRadius)
                    value = maxRadius;
                if (value < MinRadius)
                    value = MinRadius;
                if (value != mRadius)
                {
                    mRadius = value;
                    mBitmapInvalid = true;
                    Invalidate();
                }
                UpdateSize();
            }
        }

        [
            Category("Appearance"),
            Description("Indicates how ticks appear on the dial.")
        ]
        public TickStyle TickStyle
        {
            get { return mTickStyle; }
            set
            {
                if (value != mTickStyle)
                {
                    mTickStyle = value;
                    ClearTicks();
                    SetTicks(value);
                    mBitmapInvalid = true;
                    Invalidate();
                }
            }
        }

        [
            Category("Behavior"),
            Description("The current angle of the pointer. The " +
                        "angle is in tenths of a degree clockwise, and 0 " +
                        "is at the bottom of the dial.")
        ]
        public int Angle
        {
            get { return PosToAngle(mValue); }
            set { SetAngleParams(value, mMinimumAngle, mMaximumAngle); }
        }

        [Category("Behavior")]
        public int DefaultValue
        {
            get { return mDefaultValue; }
            set
            {
                if (value != mDefaultValue)
                {
                    // Room for future extensions,
                    // like showing an extra tick style.

                    mDefaultValue = value;
                }
            }
        }

        [Category("Behavior")]
        public int LargeChange
        {
            get { return mLargeChange; }
            set
            {
                if (value < mSmallChange + 1)
                    value = mSmallChange + 1;
                mLargeChange = value;
            }
        }

        [Category("Behavior")]
        public int Maximum
        {
            get { return mMaximum; }
            set
            {
                SetParams(mValue, mMinimum, value);
            }
        }

        [Category("Behavior")]
        public int MaximumAngle
        {
            get { return mMaximumAngle; }
            set { SetAngleParams(PosToAngle(mValue), mMinimumAngle, value); }
        }

        [Category("Behavior")]
        public int Minimum
        {
            get { return mMinimum;}
            set { SetParams(mValue, value, mMaximum); }
        }

        [Category("Behavior")]
        public int MinimumAngle
        {
            get { return mMinimumAngle; }
            set { SetAngleParams(PosToAngle(mValue), value, mMaximumAngle); }
        }

        [Category("Behavior")]
        public int Value
        {
            get { return mValue; }
            set { SetParams(value, mMinimum, mMaximum); }
        }

        [Category("Behavior")]
        public int RepeatDelay
        {
            get { return mRepeatDelay; }
            set { mRepeatDelay = value; }
        }

        [Category("Behavior")]
        public int RepeatRate
        {
            get { return mRepeatRate; }
            set { mRepeatRate = value; }
        }

        [Category("Behavior"), Description("The number of positions the " +
         "pointer moves in response to keyboard input (arrow keys)")]
        public int SmallChange
        {
            get { return mSmallChange; }
            set
            {
                if (value > mLargeChange)
                    value = mLargeChange / 2;
                if (value < 1)
                    value = 1;
                mSmallChange = value;
            }
        }

        [Category("Misc")]
        public Bitmap Bitmap { get { return mBitmap; } }

        [Category("Misc")]
        public Point Center { get { return new Point(mSize / 2, mSize / 2); } }

        [Category("Action")]
        public event EventHandler ValueChanged;

        [Category("Behavior")]
        public event RadioDrawEventHandler DrawPointer;
        #endregion

    }
}

