{                                                                           }
{ File:       TaskIco.pas                                                   }
{ Function:   TTaskBarIcon component                                        }
{             TTaskbarIcon is an invisible component, which enables the     }
{             programmer to put an icon in the Taskbar Notification Area,   }
{             this is the area in the Windows taskbar where the clock and   }
{             some other icons reside.                                      }
{ Language:   Delphi                                                        }
{ Author:     Rudy Velthuis (rvelthuis@gmx.de)                              }
{ Copyright:  (c) 1998,2002 Rudolph Velthuis                                }
{ Disclaimer: This code is freeware. All rights are reserved.               }
{             This code is provided as is, expressly without a              }
{             warranty of any kind. You use it at your own risk.            }
{                                                                           }
{             If you use this code, please credit me.                       }
{                                                                           }

unit TaskIco;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ShellAPI;

type
  TCustomTaskbarIcon = class(TComponent)
  private
    FEnabled: Boolean;
    FHint: string;
    FIcon: TIcon;
    FHandle: HWnd;
    FOnClick: TNotifyEvent;
    FOnDblClick: TNotifyEvent;
    FOnRightClick: TNotifyEvent;
    FOnMouseMove: TNotifyEvent;
    FWMTaskbarCreated: UINT;
    procedure SetEnabled(Value: Boolean);
    procedure SetHint(Value: string);
    procedure SetIcon(Value: TIcon);
    procedure PrivateWndProc(var Message: TMessage);
    function AppHook(var Message: TMessage): Boolean;
  protected
    function AddIcon: Boolean; dynamic;
    function DeleteIcon: Boolean; dynamic;
    function ModifyIcon(Aspect: Integer): Boolean; dynamic;
    function DoIcon(Action: DWORD; Aspect: UINT): Boolean; dynamic;
    procedure WndProc(var Message: TMessage); dynamic;
    property Handle: HWnd read FHandle;
    property Enabled: Boolean read FEnabled write SetEnabled;
    property Hint: string read FHint write SetHint;
    property Icon: TIcon read FIcon write SetIcon;
    property OnClick: TNotifyEvent read FOnClick write FOnClick;
    property OnDblClick: TNotifyEvent read FOnDblClick write FOnDblClick;
    property OnRightClick: TNotifyEvent read FOnRightClick write FOnRightClick;
    property OnMouseMove: TNotifyEvent read FOnMouseMove write FOnMouseMove;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  end;

  TTaskbarIcon = class(TCustomTaskbarIcon)
  published
    property Enabled;
    property Hint;
    property Icon;
    property OnClick;
    property OnDblClick;
    property OnRightClick;
    property OnMouseMove;
  end;

  TCustomMultiTaskbarIcon = class(TCustomTaskbarIcon)
  private
    FImageIndex: Integer;
    FImageList: TImageList;
    procedure SetImageList(const Value: TImageList);
    procedure SetImageIndex(const Value: Integer);
    procedure UpdateImage;
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    property ImageIndex: Integer read FImageIndex write SetImageIndex;
    property ImageList: TImageList read FImageList write SetImageList;
  end;

  TMultiTaskbarIcon = class(TCustomMultiTaskbarIcon)
  published
    property ImageIndex;
    property ImageList;
    property Enabled;
    property Hint;
    property OnClick;
    property OnDblClick;
    property OnRightClick;
    property OnMouseMove;
  end;

procedure Register;

implementation

const
  WM_TASKICON = WM_USER;

procedure Register;
begin
  RegisterComponents('Win32', [TTaskbarIcon]);
  RegisterComponents('Win32', [TMultiTaskbarIcon]);
end;

{ TCustomTaskbarIcon }

constructor TCustomTaskbarIcon.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FHandle := AllocateHWnd(PrivateWndProc);
  FWMTaskBarCreated := RegisterWindowMessage('TaskbarCreated');
  Application.HookMainWindow(AppHook);
  FIcon := TIcon.Create;
end;

destructor TCustomTaskbarIcon.Destroy;
begin
  Enabled := False;
  Application.UnhookMainWindow(AppHook);
  FIcon.Free;
  if FHandle <> 0 then
    DeallocateHwnd(FHandle);
  inherited Destroy;
end;

procedure TCustomTaskbarIcon.PrivateWndProc(var Message: TMessage);
begin
  WndProc(Message);
end;

procedure TCustomTaskbarIcon.SetEnabled(Value: Boolean);
begin
  if Value <> FEnabled then
  begin
    FEnabled := Value;
    if Value then
      AddIcon
    else
      DeleteIcon;
  end;
end;

procedure TCustomTaskbarIcon.SetHint(Value: string);
begin
  FHint := Value;
  ModifyIcon(NIF_TIP);
end;

procedure TCustomTaskbarIcon.SetIcon(Value: TIcon);
begin
  FIcon.Assign(Value);
  ModifyIcon(NIF_ICON);
end;

function TCustomTaskbarIcon.DoIcon(Action: DWORD; Aspect: UINT): Boolean;
var
  Data: TNotifyIconData;
begin
  with Data do
  begin
    cbSize := SizeOf(Data);
    wnd := FHandle;
    uID := 0;
    uFlags := Aspect or NIF_MESSAGE;
    uCallbackMessage := WM_TASKICON;
    if Aspect and NIF_ICON <> 0 then
      if FIcon.Handle <> 0 then
        hIcon := FIcon.Handle
      else
        hIcon := LoadIcon(0, IDI_WINLOGO);
    if Aspect and NIF_TIP <> 0 then
      StrLCopy(szTip, PChar(FHint), 63);
  end;
  Result := Shell_NotifyIcon(Action, @Data);
end;

function TCustomTaskbarIcon.AddIcon: Boolean;
begin
  Result := DoIcon(NIM_ADD, NIF_TIP or NIF_ICON);
end;

function TCustomTaskbarIcon.ModifyIcon(Aspect: Integer): Boolean;
begin
  if FEnabled then
    Result := DoIcon(NIM_MODIFY, Aspect)
  else
    Result := False;
end;

function TCustomTaskbarIcon.DeleteIcon: Boolean;
begin
  Result := DoIcon(NIM_DELETE, 0);
end;

procedure TCustomTaskbarIcon.WndProc(var Message: TMessage);
begin
  with Message do
  begin
    if Msg = WM_TASKICON then
    case LParam of
      WM_LBUTTONUP:
        if Assigned(FOnClick) then FOnClick(Self);
      WM_LBUTTONDBLCLK:
        if Assigned(FOnDblClick) then FOnDblClick(Self);
      WM_RBUTTONUP:
        if Assigned(FOnRightClick) then FOnRightClick(Self);
      WM_MOUSEMOVE:
        if Assigned(FOnMouseMove) then FOnMouseMove(Self);
    end
    else
      Result := DefWindowProc(Handle, Msg, WParam, LParam);
  end;
end;

function TCustomTaskbarIcon.AppHook(var Message: TMessage): Boolean;
begin
  Result := Message.Msg = FWMTaskbarCreated;
  if Result then
    AddIcon;
end;

{ TCustomMultiTaskbarIcon }

procedure TCustomMultiTaskbarIcon.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited;
  if (Operation = opRemove) and (AComponent = FImageList) then
    SetImageList(nil);
end;

procedure TCustomMultiTaskbarIcon.SetImageIndex(const Value: Integer);
begin
  if (Value <> FImageIndex) and
     Assigned(FImageList) and (Value < FImageList.Count) then
  begin
    FImageIndex := Value;
    UpdateImage;
  end;
end;

procedure TCustomMultiTaskbarIcon.SetImageList(const Value: TImageList);
begin
  if Value <> FImageList then
  begin
    FImageList := Value;
    UpdateImage;
  end;
end;

procedure TCustomMultiTaskbarIcon.UpdateImage;
var
  I: TIcon;
begin
  if (FImageList = nil) or
     (FImageIndex < 0) or
     (FImageIndex >= FImageList.Count) then
  begin
    Enabled := False;
    Exit;
  end;
  I := TIcon.Create;
  try
    FImageList.GetIcon(FImageIndex, I);
    Icon := I;
  finally
    I.Free;
  end;
end;

end.
